/*
#
# COPYRIGHT INFORMATION
#
# Copyright (C) 2013 Indian Institute of Science
# Bangalore 560012, INDIA
#
# This program is part of the CODD Metadata Processor
# software distribution invented at the Database Systems Lab,
# Indian Institute of Science. The use of the software is governed
# by the licensing agreement set up between the copyright owner,
# Indian Institute of Science, and the licensee.
#
# This program is distributed WITHOUT ANY WARRANTY;
# without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
# The public URL of the CODD project is
# http://dsl.serc.iisc.ernet.in/projects/CODD/index.html
#
# For any issues, contact
# Prof. Jayant R. Haritsa
# SERC
# Indian Institute of Science
# Bangalore 560012, India.
# 

# Email: haritsa@dsl.serc.iisc.ernet.in
# 
#
 */
package iisc.dsl.codd.client;

import iisc.dsl.codd.DatalessException;
import iisc.dsl.codd.db.ColumnStatistics;
import iisc.dsl.codd.db.DBConstants;
import iisc.dsl.codd.db.Database;
import iisc.dsl.codd.db.IndexStatistics;
import iisc.dsl.codd.db.RelationStatistics;
import iisc.dsl.codd.db.db2.DB2Database;
import iisc.dsl.codd.db.mssql.MSSQLDatabase;
import iisc.dsl.codd.db.nonstopsql.NonStopSQLDatabase;
import iisc.dsl.codd.db.oracle.OracleDatabase;
import iisc.dsl.codd.db.postgres.PostgresDatabase;
import iisc.dsl.codd.db.sybase.SybaseDatabase;
import iisc.dsl.codd.ds.Constants;
import iisc.dsl.codd.ds.DBSettings;
import iisc.dsl.codd.ds.Statistics;

import java.io.File;
import java.io.FileOutputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JOptionPane;

/**
 * This class implements the Transfer mode of CODD Metadata Processor.
 * The user can either save the metadata into a file or do a transfer to a running database instance.
 * Inter / Intra Engine transfer is decided based on the chosen connection parameters.
 * @author dsladmin
 */
public class TransferMode extends javax.swing.JFrame {

	/**
	 * Generated serialVersionUID
	 */
	private static final long serialVersionUID = -3209727075202307671L;
	String type;  // DB Engine type
	String[] relations; // Selected Relations for transfer
	Database database; // Source Database object
	Database destDatabase; // Destination Database object
	String destType; // Destination DB Engie Type
	File selectedFile; // to writeMetadataStatistics
	String emptyConfig; // Empty line
	HashMap<String, String> connectionParameters; // HashMap for Stored Connection Parameters
	
	public static int mode;

	/**
	 * Constructor for TransferMode
	 * @param Type - Connected Database Vendor name
	 * @param relations - Array of selected relations
	 * @param database - Database object (source)
	 */
	public TransferMode(String Type, String[] relations, Database database) {
		type = Type;
		this.relations = relations;
		this.database = database;
		this.destType = null;
		this.destDatabase = null;
		this.selectedFile = null;
		this.mode = 0;
		initComponents();

		setLocationRelativeTo(null);
		setTitle("TransferMode - " + type);
		emptyConfig = new String("");

		// Disable all the components except the three radio buttons.
		// Database Transfer
		jDatabaseConnParams.setEnabled(false);   // Dest Database Connection Parameters
		jComboBoxConnectionParameters.setEnabled(false); // Dest Database Connection Info

		// Save - File Transfer
		jLabel3.setEnabled(false);   // Enter Directory Path
		jTextField2.setEnabled(false);   // Directory Path
		jTextField2.setText("");   // Directory Path
		jButton2.setEnabled(false);   // Browse
		
		if(type.equals(DBConstants.NONSTOPSQL)) {
			jRadioButton2.setEnabled(false);
	    }

		// Populate Connection parameter ComboBox
		jComboBoxConnectionParameters.removeAllItems();
		jComboBoxConnectionParameters.addItem(emptyConfig);
		connectionParameters = DBConstants.readStoredConnectionParameters();
		if (connectionParameters != null) {
			Set s = connectionParameters.keySet();
			Iterator i = s.iterator();
			while (i.hasNext()) {
				String key = (String) i.next();
				jComboBoxConnectionParameters.addItem(key);
			}
			jComboBoxConnectionParameters.setSelectedItem(emptyConfig);
		} else {
			jComboBoxConnectionParameters.removeAllItems();
		}
	}

	/** This method is called from within the constructor to
	 * initialize the form.
	 * WARNING: Do NOT modify this code. The content of this method is
	 * always regenerated by the Form Editor.
	 */
	@SuppressWarnings("unchecked")
	// <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
	private void initComponents() {

		buttonGroup1 = new javax.swing.ButtonGroup();
		jCheckBox1 = new javax.swing.JCheckBox();
		jPanel1 = new iisc.dsl.codd.client.gui.BaseJPanel("img/bg_net.png");
		jPanel2 = new javax.swing.JPanel();
		jRadioButton2 = new javax.swing.JRadioButton();
		jSeparator2 = new javax.swing.JSeparator();
		jDatabaseConnParams = new javax.swing.JLabel();
		jComboBoxConnectionParameters = new javax.swing.JComboBox();
		jRadioButton3 = new javax.swing.JRadioButton();
		jLabel3 = new javax.swing.JLabel();
		jTextField2 = new javax.swing.JTextField();
		jButton2 = new javax.swing.JButton();
		jButton3 = new javax.swing.JButton();
		jButton5 = new javax.swing.JButton();

		jCheckBox1.setText("jCheckBox1");

		setDefaultCloseOperation(javax.swing.WindowConstants.EXIT_ON_CLOSE);
		setResizable(false);

		jPanel2.setBorder(javax.swing.BorderFactory.createTitledBorder(null, "Select Mode", javax.swing.border.TitledBorder.DEFAULT_JUSTIFICATION, javax.swing.border.TitledBorder.DEFAULT_POSITION, new java.awt.Font("Cambria", 0, 14), new java.awt.Color(0, 102, 204))); // NOI18N
		jPanel2.setOpaque(false);

		buttonGroup1.add(jRadioButton2);
		jRadioButton2.setFont(new java.awt.Font("Cambria", 0, 14)); // NOI18N
		jRadioButton2.setText("Inter Engine Transfer / Intra EngineTransfer");
		jRadioButton2.setToolTipText("Tranfer Metadata to same DB Engine (network machine) or Inter Engine Transfer");
		jRadioButton2.setOpaque(false);
		jRadioButton2.addActionListener(new java.awt.event.ActionListener() {
			public void actionPerformed(java.awt.event.ActionEvent evt) {
				jRadioButton2ActionPerformed(evt);
			}
		});

		jDatabaseConnParams.setFont(new java.awt.Font("Cambria", 0, 14)); // NOI18N
		jDatabaseConnParams.setText("Database Connection Properties:");

		jComboBoxConnectionParameters.setFont(new java.awt.Font("Cambria", 0, 14)); // NOI18N

		buttonGroup1.add(jRadioButton3);
		jRadioButton3.setFont(new java.awt.Font("Cambria", 0, 14)); // NOI18N
		jRadioButton3.setText("Save Metadata into a file");
		jRadioButton3.setToolTipText("Saves metadat in native format to a file");
		jRadioButton3.setOpaque(false);
		jRadioButton3.addActionListener(new java.awt.event.ActionListener() {
			public void actionPerformed(java.awt.event.ActionEvent evt) {
				jRadioButton3ActionPerformed(evt);
			}
		});

		jLabel3.setFont(new java.awt.Font("Cambria", 0, 14)); // NOI18N
		jLabel3.setText("Enter the directory path to save the metadata files:");
		jLabel3.setToolTipText("");

		jTextField2.setFont(new java.awt.Font("Cambria", 0, 14)); // NOI18N

		jButton2.setFont(new java.awt.Font("Cambria", 0, 14)); // NOI18N
		jButton2.setText("Browse");
		jButton2.addActionListener(new java.awt.event.ActionListener() {
			public void actionPerformed(java.awt.event.ActionEvent evt) {
				jButton2ActionPerformed(evt);
			}
		});

		javax.swing.GroupLayout jPanel2Layout = new javax.swing.GroupLayout(jPanel2);
		jPanel2.setLayout(jPanel2Layout);
		jPanel2Layout.setHorizontalGroup(
				jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
				.addGroup(jPanel2Layout.createSequentialGroup()
						.addContainerGap()
						.addGroup(jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
								.addComponent(jSeparator2)
								.addGroup(jPanel2Layout.createSequentialGroup()
										.addComponent(jDatabaseConnParams)
										.addGap(18, 18, 18)
										.addComponent(jComboBoxConnectionParameters, 0, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
										.addGroup(jPanel2Layout.createSequentialGroup()
												.addGroup(jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
														.addGroup(javax.swing.GroupLayout.Alignment.TRAILING, jPanel2Layout.createSequentialGroup()
																.addComponent(jTextField2)
																.addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
																.addComponent(jButton2, javax.swing.GroupLayout.PREFERRED_SIZE, 120, javax.swing.GroupLayout.PREFERRED_SIZE))
																.addComponent(jRadioButton3)
																.addComponent(jRadioButton2, javax.swing.GroupLayout.PREFERRED_SIZE, 508, javax.swing.GroupLayout.PREFERRED_SIZE)
																.addComponent(jLabel3))
																.addGap(0, 0, Short.MAX_VALUE)))
																.addContainerGap())
				);
		jPanel2Layout.setVerticalGroup(
				jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
				.addGroup(jPanel2Layout.createSequentialGroup()
						.addContainerGap()
						.addComponent(jRadioButton2)
						.addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
						.addGroup(jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.CENTER)
								.addComponent(jDatabaseConnParams)
								.addComponent(jComboBoxConnectionParameters, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
								.addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
								.addComponent(jSeparator2, javax.swing.GroupLayout.PREFERRED_SIZE, 10, javax.swing.GroupLayout.PREFERRED_SIZE)
								.addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
								.addComponent(jRadioButton3)
								.addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
								.addComponent(jLabel3)
								.addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
								.addGroup(jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
										.addComponent(jTextField2, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
										.addComponent(jButton2, javax.swing.GroupLayout.PREFERRED_SIZE, 40, javax.swing.GroupLayout.PREFERRED_SIZE))
										.addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
				);

		jButton3.setText("<< Back");
		jButton3.addActionListener(new java.awt.event.ActionListener() {
			public void actionPerformed(java.awt.event.ActionEvent evt) {
				jButton3ActionPerformed(evt);
			}
		});

		jButton5.setText("Continue >>");
		jButton5.addActionListener(new java.awt.event.ActionListener() {
			public void actionPerformed(java.awt.event.ActionEvent evt) {
				jButton5ActionPerformed(evt);
			}
		});

		javax.swing.GroupLayout jPanel1Layout = new javax.swing.GroupLayout(jPanel1);
		jPanel1.setLayout(jPanel1Layout);
		jPanel1Layout.setHorizontalGroup(
				jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
				.addGroup(jPanel1Layout.createSequentialGroup()
						.addContainerGap()
						.addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
								.addComponent(jPanel2, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
								.addGroup(jPanel1Layout.createSequentialGroup()
										.addComponent(jButton3, javax.swing.GroupLayout.PREFERRED_SIZE, 120, javax.swing.GroupLayout.PREFERRED_SIZE)
										.addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
										.addComponent(jButton5, javax.swing.GroupLayout.PREFERRED_SIZE, 120, javax.swing.GroupLayout.PREFERRED_SIZE)))
										.addContainerGap())
				);
		jPanel1Layout.setVerticalGroup(
				jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
				.addGroup(jPanel1Layout.createSequentialGroup()
						.addContainerGap()
						.addComponent(jPanel2, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
						.addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
						.addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
								.addComponent(jButton3, javax.swing.GroupLayout.PREFERRED_SIZE, 40, javax.swing.GroupLayout.PREFERRED_SIZE)
								.addComponent(jButton5, javax.swing.GroupLayout.PREFERRED_SIZE, 40, javax.swing.GroupLayout.PREFERRED_SIZE))
								.addContainerGap(19, Short.MAX_VALUE))
				);

		getContentPane().add(jPanel1, java.awt.BorderLayout.CENTER);

		pack();
	}// </editor-fold>//GEN-END:initComponents

	private void jButton3ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButton3ActionPerformed
		// Back Button
		new ModeSelection(database, relations).setVisible(true);
		this.dispose();
	}//GEN-LAST:event_jButton3ActionPerformed

	/**
	 * Transfers the Metadata from source to destination database.
	 *
	 * For all relations
	 *    Get & Update relation information
	 *    For all Columns in this relation
	 *      Get & Update column info
	 *    For all Indexed in this relation
	 *      Get & Update index info
	 *
	 * @param source - Source Database Object
	 * @param dest - Destination Database Object
	 * @param isSave - Boolean variable indicating to save the metadata into a file.
	 */
	private void transfer(Database source, Database dest, boolean isSave) throws DatalessException {
		String path = null;
		boolean success = true;
		if (isSave && selectedFile == null) {
			JOptionPane.showMessageDialog(null, "Error: Destination folder to save metadata is not chosen.", "CODD - Exception", JOptionPane.ERROR_MESSAGE);
			return;
		}

		if (!isSave && source.getSettings().getDbVendor().equals(DBConstants.MSSQL) && dest.getSettings().getDbVendor().equals(DBConstants.MSSQL)) {
			// MSSQL Inter Engine Transfer. We have implemented transfer function only for MSSQL.
			try {
				success = source.transfer(relations, dest);
			} catch (DatalessException e) {
				JOptionPane.showMessageDialog(null, "Exception Caught: Could not transfer metadata statistics.", "CODD - Exception", JOptionPane.ERROR_MESSAGE);
				success = false;
			}
		} else {
			if (isSave) {
				String allDBInfo = source.getSettings().getDbVendor() + "-" + source.getSettings().getSchema() + "-" + source.getSettings().getDbName();
				path = selectedFile.getAbsolutePath() + Constants.PathSeparator + "CODD_Statistics-" + allDBInfo + Constants.PathSeparator;
				File f = new File(path);
				if (!f.exists()) {
					f.mkdir();
				}
				/*
				 * Write a file to identify the database type for which we write metadata.
				 */
				try {
					String filename = path + "dbType";
					FileOutputStream fos = new FileOutputStream(filename);
					ObjectOutputStream oos = new ObjectOutputStream(fos);
					oos.writeObject(source.getSettings().getDbVendor());
					oos.flush();
					oos.close();
					/*
                    FileWriter fw = new FileWriter(filename);
                    BufferedWriter out = new BufferedWriter(fw);
                    out.write(source.getSettings().getDbVendor() +"\n");
                    out.close();
					 *
					 */
				} catch (Exception e) {
					Constants.CPrintErrToConsole(e);
					success = false;
				}
			}
			/*
            boolean colStatHistSetNull = false;
            boolean other2Orcl = false;
            if (!type.equals(DBConstants.ORACLE) && dest != null && dest.getSettings().getDbVendor().equals(DBConstants.ORACLE)) {
            other2Orcl = true;
            int status = JOptionPane.showConfirmDialog(null, "You have chosen other DB engine statistics to transfer to Oracle. "
            + "Clicking on Yes will use input histogram column values as Oracle height balanced histogram end points, No will not update Oracle Histogram.", "Choose the option.", JOptionPane.YES_NO_OPTION);
            if (status == JOptionPane.NO_OPTION) {
            colStatHistSetNull = true;
            }
            }
			 */
			if (!isSave && (dest.getSettings().getDbVendor().equals(DBConstants.ORACLE) || dest.getSettings().getDbVendor().equals(DBConstants.POSTGRES))) {
				Constants.status = -1; // Mapping warning
			}
			for (int i = 0; i < relations.length; i++) {
				String relationName = relations[i];
				try {
					String TransferOrSave;
					if (isSave) {
						TransferOrSave = "Saving";
					} else {
						TransferOrSave = "Transferring";
					}
					
					Statistics stats = fetchStatisics(relationName, source, dest, TransferOrSave, isSave);

					if (isSave) {
						String filename = path + relations[i];
						// Write one single Object for this relation
						FileOutputStream fos = new FileOutputStream(filename);
						ObjectOutputStream oos = new ObjectOutputStream(fos);
						oos.writeObject(stats);
						oos.flush();
						oos.close();
						//stats.writeStatisticsToFile(filename+".txt"); // No need write the statisics in user readable format, as this can't be read by the tool.
					}
				} catch (Exception e) {
					Constants.CPrintErrToConsole(e);
					JOptionPane.showMessageDialog(null, "Exception Caught: Could not transfer metadata object for " + relationName + ".", "CODD - Exception", JOptionPane.ERROR_MESSAGE);
					success = false;
				}
			}  // End-For relations.
			if (!isSave && (dest.getSettings().getDbVendor().equals(DBConstants.ORACLE) || dest.getSettings().getDbVendor().equals(DBConstants.POSTGRES))) {
				Constants.status = 0; // Reset Mapping warning
			}
		} // End-else of if (MSSQL Inter transfer)
		if (success) {
			JOptionPane.showMessageDialog(null, "Statistics Transfer of Relations is Successful.", "CODD - Message", JOptionPane.INFORMATION_MESSAGE);
			new GetRelationFrame(database).setVisible(true);
			//new ModeSelection(database, passedRelation).setVisible(true);
			this.dispose();
		}
	}

	public static Statistics fetchStatisics(String relationName, Database source, Database dest, String TransferOrSave, boolean isSave) throws DatalessException {
		Statistics stats = new Statistics(relationName, source.getSettings().getDbVendor());
		Constants.CPrintToConsole(TransferOrSave + " Relation Statistics for : " + relationName, Constants.DEBUG_FIRST_LEVEL_Information);
		RelationStatistics relStat = source.getRelationStatistics(relationName);
		if (isSave) {
			stats.setRelationStatistics(relStat);
		} else {
			dest.setRelationStatistics(relationName, relStat);
		}

		String[] attribs = source.getAttributes(relationName);
		Constants.CPrintToConsole(TransferOrSave + " Attribute statistics: ", Constants.DEBUG_FIRST_LEVEL_Information);
		for (int j = 0; j < attribs.length; j++) {
			mode = 1; 											/* NONSTOP - This is used in Construct Mode using Baseline Configuration */
			String type = source.getType(relationName, attribs[j]);
			ColumnStatistics colStat = source.getColumnStatistics(relationName, attribs[j], relStat.getCardinality());
			Constants.CPrintToConsole("\t " + TransferOrSave + " attribute " + attribs[j] + " : Type - " + type + " : Constraint - " + colStat.getConstraint(), Constants.DEBUG_FIRST_LEVEL_Information);
			if (isSave) {
				// Column Statistics
				stats.addColumnStatistics(attribs[j], colStat);
			} else {
				/*
                if (other2Orcl && colStatHistSetNull) {
                colStat.setHistogram(null);
                } */
				dest.setColumnStatistics(relationName, attribs[j], colStat, relStat.getCardinality());
			}
		}

		Constants.CPrintToConsole(TransferOrSave + " for Indexed Attributes: ", Constants.DEBUG_FIRST_LEVEL_Information);
		Map map = source.getIndexedColumns(relationName);
		Iterator iterator = map.keySet().iterator();
		while (iterator.hasNext()) {
			String key = (String) iterator.next();
			String colNames = (String) map.get(key);
			ArrayList<String> cols = new ArrayList();
			if (colNames.contains("+")) {
				String[] col = colNames.split(Pattern.quote("+"));
				for (int c = 0; c < col.length; c++) {
					cols.add(col[c]);
				}
			} else {
				cols.add(colNames);
			}
			IndexStatistics indexStat = source.getIndexStatistics(relationName, cols);

			Constants.CPrintToConsole("\t " + TransferOrSave + " for index " + key + " : " + map.get(key), Constants.DEBUG_FIRST_LEVEL_Information);

			if (isSave) {
				// Index Statistics
				stats.addIndexStatistics(colNames, indexStat);
			} else {
				dest.setIndexStatistics(relationName, cols, indexStat);
			}
		}
		return stats;
	}
	
	
	private void jButton5ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButton5ActionPerformed
		// Continue Button
		/**
		 * Determine the option chosen.
		 *  Transfer to database.
		 *      Obtain the database connection and call transfer with noSave.
		 *  Save
		 *      Obtain the file directory and call transfer with save.
		 *
		 */
		if (jRadioButton2.isSelected()) {
			// Destination Database is up and running. Read from source and transfer to destination database.
			String connectionInfo = (String) jComboBoxConnectionParameters.getSelectedItem();
			if (connectionInfo != null && !connectionInfo.equals(emptyConfig)) {
				String dbServerName, dbType, dbServerPort, dbName, dbSchema, dbUserName, dbPassword, serverInstance;
				String params = connectionParameters.get(jComboBoxConnectionParameters.getSelectedItem());
				String[] temp = params.split("%%");
				if (temp.length != 7 && temp.length != 8) {
					JOptionPane.showMessageDialog(null, "Error in reading configuration.", "CODD - Error", JOptionPane.ERROR_MESSAGE);
				} else {
					dbServerName = temp[0];
					dbType = temp[1];
					dbServerPort = temp[2];
					dbName = temp[3];
					dbSchema = temp[4];
					dbUserName = temp[5];
					dbPassword = temp[6];
					if (dbType.equals(DBConstants.MSSQL)) {
						serverInstance = temp[7];
					} else {
						serverInstance = emptyConfig;
					}
					if (dbType.equals(DBConstants.MSSQL) && !database.getSettings().getDbVendor().equals(DBConstants.MSSQL)) {
						JOptionPane.showMessageDialog(null, "Inter Engine transfer to SQL Server is not supported.You are required to select any other destination database to move ahead. To abort transfer mode press Exit.", "CODD Error Message - Action Required", 0);
					} else {
						try {
							DBSettings settings = new DBSettings(dbServerName, dbServerPort, dbType, dbName, dbSchema, dbUserName, dbPassword);
							settings.setSqlServerInstanceName(serverInstance);

							Database destDatabase = null;
							if (dbType.equals(DBConstants.DB2)) {
								destDatabase = (Database) new DB2Database(settings);
							} else if (dbType.equals(DBConstants.MSSQL)) {
								destDatabase = (Database) new MSSQLDatabase(settings);
							} else if (dbType.equals(DBConstants.ORACLE)) {
								destDatabase = (Database) new OracleDatabase(settings);
							} else if (dbType.equals(DBConstants.POSTGRES)) {
								destDatabase = (Database) new PostgresDatabase(settings);
							} else if (dbType.equals(DBConstants.SYBASE)) {
								destDatabase = (Database) new SybaseDatabase(settings);
							}

							//-------------------------------------------------------------------------------------------------------------------


							if (destDatabase == null) {
								JOptionPane.showMessageDialog(null, "Exception Caught: Could not get database object.", "CODD - Exception", JOptionPane.ERROR_MESSAGE);
							} else if (destDatabase.connect()) {
								Constants.CPrintToConsole("Destination Databae Connection established. ", Constants.DEBUG_FIRST_LEVEL_Information);
								if (!destDatabase.stopAutoUpdateOfStats()) {
									JOptionPane.showMessageDialog(null, "Exception Caught: Could not able to stop auto maintanance of statistics.", "CODD - Exception", JOptionPane.ERROR_MESSAGE);
								} else {
									transfer(database, destDatabase, false);
								}
							}
						} catch (Exception e) {
							Constants.CPrintErrToConsole(e);
							JOptionPane.showMessageDialog(null, "Exception Caught: Could not able to make connection to " + connectionInfo + ".", "CODD - Exception", JOptionPane.ERROR_MESSAGE);
						}
					}
				}
			} else {
				JOptionPane.showMessageDialog(null, "You are required to select any one of the destination database to move ahead. To abort transfer mode press Exit.", "CODD Error Message - Action Required", 0);
			}
		} else if ((jRadioButton3.isSelected())) {
			try {
				// File Transfer is chosen.
				transfer(database, destDatabase, true);
			} catch (DatalessException ex) {
				Constants.CPrintErrToConsole(ex);
				JOptionPane.showMessageDialog(null, "Dataless Exception.", "CODD - Exception", JOptionPane.ERROR_MESSAGE);
			}
		} else {
			JOptionPane.showMessageDialog(null, "You are required to select one of the two option to move ahead. To abort transfer mode press Exit.", "CODD Error Message - Action Required", 0);
		}

	}//GEN-LAST:event_jButton5ActionPerformed

	private void jRadioButton2ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jRadioButton2ActionPerformed
		// Database Transfer
		jDatabaseConnParams.setEnabled(true);   // Dest Database Connection Parameters
		jComboBoxConnectionParameters.setEnabled(true); // Dest Database Connection Info

		// File Transfer
		jLabel3.setEnabled(false);   // Enter Directory Path
		jTextField2.setEnabled(false);   // Directory Path
		jTextField2.setText("");   // Directory Path
		jButton2.setEnabled(false);   // Browse
	}//GEN-LAST:event_jRadioButton2ActionPerformed

	private void jRadioButton3ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jRadioButton3ActionPerformed
		// Database Transfer
		jDatabaseConnParams.setEnabled(false);   // Dest Database Connection Parameters
		jComboBoxConnectionParameters.setEnabled(false); // Dest Database Connection Info

		// File Transfer is chosen
		jLabel3.setEnabled(true);   // Enter Directory Path
		jTextField2.setEnabled(true);   // Directory Path
		jTextField2.setText("");   // Directory Path
		jButton2.setEnabled(true);   // Browse
	}//GEN-LAST:event_jRadioButton3ActionPerformed

	private void jButton2ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButton2ActionPerformed
		// Get the directory in which we will write the metadata

		JFileChooser filechooser = new JFileChooser();
		filechooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
		String currDir = Constants.getWorkingDirectory();
		filechooser.setCurrentDirectory(new File(currDir));
		int returnvalue = filechooser.showOpenDialog(this);
		if (returnvalue == JFileChooser.APPROVE_OPTION) {
			selectedFile = filechooser.getSelectedFile();
			jTextField2.setText(selectedFile.getAbsolutePath());
			Constants.CPrintToConsole("Write Metadata files into this directory.: " + selectedFile.getAbsolutePath(), Constants.DEBUG_FIRST_LEVEL_Information);
		}

	}//GEN-LAST:event_jButton2ActionPerformed
	private static JFrame dbFrame;
	// Variables declaration - do not modify//GEN-BEGIN:variables
	private javax.swing.ButtonGroup buttonGroup1;
	private javax.swing.JButton jButton2;
	private javax.swing.JButton jButton3;
	private javax.swing.JButton jButton5;
	private javax.swing.JCheckBox jCheckBox1;
	private javax.swing.JComboBox jComboBoxConnectionParameters;
	private javax.swing.JLabel jDatabaseConnParams;
	private javax.swing.JLabel jLabel3;
	private javax.swing.JPanel jPanel1;
	private javax.swing.JPanel jPanel2;
	private javax.swing.JRadioButton jRadioButton2;
	private javax.swing.JRadioButton jRadioButton3;
	private javax.swing.JSeparator jSeparator2;
	private javax.swing.JTextField jTextField2;
	// End of variables declaration//GEN-END:variables
}