package iisc.dsl.coddgen.ui;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;

import javax.swing.AbstractCellEditor;
import javax.swing.Action;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumnModel;

/**
 *  The CheckBoxColumn class provides a renderer and an editor that looks like a
 *  JCheckBox. The renderer and editor will then be used for a specified column
 *  in the table. The TableModel will contain the boolean value to be displayed on
 *  the check box.
 *
 *  When the button is invoked the provided Action is invoked. The
 *  source of the Action will be the table. The action command will contain
 *  the model row number of the button that was clicked.
 *
 * @cite Adapted from ButtonColumn by Rob Camick https://tips4java.wordpress.com/2009/07/12/table-button-column/
 */
public class CheckBoxColumn extends AbstractCellEditor implements TableCellRenderer, TableCellEditor, ActionListener, MouseListener {
    private final JTable        table;
    private final Action        action;

    private static final JLabel EMPTY_JLABEL = new JLabel();

    private final JCheckBox     renderButton;
    private final JCheckBox     editButton;
    private Boolean             editorValue;
    private boolean             isButtonColumnEditor;

    /**
     *  Create the ButtonColumn to be used as a renderer and editor. The
     *  renderer and editor will automatically be installed on the TableColumn
     *  of the specified column.
     *
     *  @param table the table containing the button renderer/editor
     *  @param action the Action to be invoked when the button is invoked
     *  @param column the column to which the button renderer/editor is added
     */
    public CheckBoxColumn(JTable table, Action action, int column) {
        this.table = table;
        this.action = action;

        renderButton = new JCheckBox();
        editButton = new JCheckBox();
        editButton.setFocusPainted(false);
        editButton.addActionListener(this);

        TableColumnModel columnModel = table.getColumnModel();
        columnModel.getColumn(column).setCellRenderer(this);
        columnModel.getColumn(column).setCellEditor(this);
        table.addMouseListener(this);
    }

    @Override
    public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) {
        if (value == null) {
            editorValue = null;
            return EMPTY_JLABEL;
        } else if (value instanceof Boolean) {
            editorValue = (Boolean) value;
            editButton.setSelected(editorValue);
            return editButton;
        } else {
            editorValue = null;
            return EMPTY_JLABEL;
        }
    }

    @Override
    public Boolean getCellEditorValue() {
        return editorValue;
    }

    //
    //  Implement TableCellRenderer interface
    //
    @Override
    public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {

        if (value == null)
            return EMPTY_JLABEL;
        else if (value instanceof Boolean) {
            renderButton.setSelected((Boolean) value);
            return renderButton;
        } else
            return EMPTY_JLABEL;
    }

    //
    //  Implement ActionListener interface
    //
    /*
     *  The button has been pressed. Stop editing and invoke the custom Action
     */
    @Override
    public void actionPerformed(ActionEvent e) {
        int row = table.convertRowIndexToModel(table.getEditingRow());
        fireEditingStopped();

        //  Invoke the Action

        ActionEvent event = new ActionEvent(table, ActionEvent.ACTION_PERFORMED, "" + row);
        action.actionPerformed(event);
    }

    //
    //  Implement MouseListener interface
    //
    /*
     *  When the mouse is pressed the editor is invoked. If you then then drag
     *  the mouse to another cell before releasing it, the editor is still
     *  active. Make sure editing is stopped when the mouse is released.
     */
    @Override
    public void mousePressed(MouseEvent e) {
        if (table.isEditing() && table.getCellEditor() == this) {
            isButtonColumnEditor = true;
        }
    }

    @Override
    public void mouseReleased(MouseEvent e) {
        if (isButtonColumnEditor && table.isEditing()) {
            table.getCellEditor().stopCellEditing();
        }

        isButtonColumnEditor = false;
    }

    @Override
    public void mouseClicked(MouseEvent e) {}

    @Override
    public void mouseEntered(MouseEvent e) {}

    @Override
    public void mouseExited(MouseEvent e) {}
}