package iisc.dsl.coddgen.ui;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

import com.google.gson.Gson;

import iisc.dsl.codd.DatalessException;
import iisc.dsl.codd.client.TransferMode;
import iisc.dsl.codd.db.ColumnStatistics;
import iisc.dsl.codd.db.Database;
import iisc.dsl.codd.db.HistogramObject;
import iisc.dsl.codd.db.db2.DB2FreqHistObject;
import iisc.dsl.codd.db.postgres.PostgresColumnStatistics;
import iisc.dsl.codd.ds.Constants;
import iisc.dsl.codd.ds.Statistics;
import iisc.dsl.coddgen.ui.model.ClientInputFrameOut;
import iisc.dsl.coddgen.ui.model.CollectingInputFrameOut;
import iisc.dsl.coddgen.ui.utils.ConsoleOutputStream;
import iisc.dsl.coddgen.ui.utils.Utils;
import in.ac.iisc.cds.dsl.cdgclient.constants.PostgresCConfig;
import in.ac.iisc.cds.dsl.cdgclient.constants.PostgresCConfig.Key;
import in.ac.iisc.cds.dsl.cdgclient.preprocess.ExplainAnalyzeToAlqpPostgres;
import in.ac.iisc.cds.dsl.cdgclient.preprocess.QueryToExplainAnalyzePostgres;
import in.ac.iisc.cds.dsl.cdgclient.preprocess.SchemaAnalyze;
import in.ac.iisc.cds.dsl.cdgclient.preprocess.SchemaAnalyzePostgres;
import in.ac.iisc.cds.dsl.cdgvendor.model.Alqp;
import in.ac.iisc.cds.dsl.cdgvendor.model.HistogramMappingInfo;
import in.ac.iisc.cds.dsl.cdgvendor.model.SchemaInfo;
import in.ac.iisc.cds.dsl.cdgvendor.utils.FileUtils;

@SuppressWarnings("serial")
public class CollectingInputFrame extends JFrame {

    private static final String       NEWLINE     = "\n";
    private static final DateFormat   DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd");

    private final JFrame              prevFrame;
    private final ClientInputFrameOut prevFrameOut;

    private final ConsoleOutputStream cos;
    private final JButton             backButton;
    private final JButton             nextButton;

    private CollectingInputFrameOut   out;

    public CollectingInputFrame(JFrame prevFrame, ClientInputFrameOut prevFrameOut) {

        this.prevFrame = prevFrame;
        this.prevFrameOut = prevFrameOut;

        cos = new ConsoleOutputStream();
        backButton = new JButton("Back");
        nextButton = new JButton("Next");

        Utils.setFrame(this, "Collecting Input for generation...");
        initComponents();

        new Thread(new Runnable() {
            @Override
            public void run() {
                startProcessing();
            }
        }).start();
    }

    private void initComponents() {

        JPanel ctrlPanel = Utils.newCtlrsPanel();
        backButton.setEnabled(false);
        backButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                onClickBack();
            }
        });
        ctrlPanel.add(backButton);
        nextButton.setEnabled(false);
        nextButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                onClickNext();
            }
        });
        Utils.setEnterButton(this, nextButton);
        ctrlPanel.add(nextButton);

        JPanel bgPanel = Utils.newBgPanel();
        bgPanel.setLayout(new BorderLayout());
        bgPanel.add(new JScrollPane(cos.getConsole()), BorderLayout.CENTER);
        bgPanel.add(ctrlPanel, BorderLayout.SOUTH);
        getContentPane().add(bgPanel);
        setLocationRelativeTo(null);
    }

    private void startProcessing() {
    	
//    	Setting up basic schema location
    	Map<Key, String> overlapConfig = new HashMap<>();
    	overlapConfig.put(Key.BASICSCHEMA_TARGETFILE, prevFrameOut.outputLocation+Constants.PathSeparator+"basicschema.info");
    	PostgresCConfig.overlayOnDefaultConfig(overlapConfig);
    	

        //Step 1: Fetch the database schema
        SchemaInfo BASICSCHEMA_INFO = null;
        cos.write("Fetching schema info...");
        try {
            SchemaAnalyze sa = new SchemaAnalyzePostgres();
            BASICSCHEMA_INFO = new SchemaInfo();
            BASICSCHEMA_INFO.getTableInfos().putAll(sa.explainSchema());
        } catch (Exception ex) {
            cos.write("ERROR fetching schema" + ex.toString());
            return;
        }
        try {
            BASICSCHEMA_INFO.validateAndInit();
        } catch (Exception ex) {
            cos.write("ERROR: " + ex.getMessage());
            return;
        }

        List<String> queryNames = new ArrayList<>();
        List<String> queries = new ArrayList<>();
        for (File queryFile : prevFrameOut.workloadQueryFiles) {
            String queryName = queryFile.getName().split("\\.")[0];
            String sqlQuery = FileUtils.readFileToString(queryFile.getAbsolutePath());
            queryNames.add(queryName);
            queries.add(sqlQuery);
        }
        Collections.sort(queryNames);

        //TODO: Do following processing in sorted order
        //Step 2: Get explain-analyze for all given queries
        List<Alqp> alqps = new ArrayList<>();
        if (prevFrameOut.isExecutionPlansFetch) {
            cos.write("Fetching execution plans...");
            for (int i = 0; i < queryNames.size(); i++) {
                String queryName = queryNames.get(i);
                String sqlQuery = queries.get(i);
                String ea = QueryToExplainAnalyzePostgres.fetchAlqpFromDB(sqlQuery);
                Alqp alqp = ExplainAnalyzeToAlqpPostgres.readAlqpFromString(ea, queryName);
                alqps.add(alqp);
                cos.write("\tExecuted query " + queryName);
            }
        } else {
            cos.write("Loading execution plans...");
            for (File eaFile : prevFrameOut.executionPlanFiles) {
                String queryName = eaFile.getName().split("\\.")[0];
                String ea = FileUtils.readFileToString(eaFile.getAbsolutePath());
                Alqp alqp = ExplainAnalyzeToAlqpPostgres.readAlqpFromString(ea, queryName);
                alqps.add(alqp);
                //cos.write("\tLoaded execution plan " + queryName);
            }
        }

        //Step 3: Get metadata info
        Map<String, Statistics> relnameStatsMap;
        Set<String> relnames = BASICSCHEMA_INFO.getTableInfos().keySet();
        if (prevFrameOut.isMetadataFetch) {
            cos.write("Fetching metadata info...");
            relnameStatsMap = fetchMetadata(prevFrameOut.database, relnames);
        } else {
            cos.write("Loading metadata info...");
            relnameStatsMap = loadMetadata(prevFrameOut.metadataLocation.getAbsolutePath(), relnames);
        }
        HistogramMappingInfo mappingInfo = getAsHistogramMappingInfo(relnameStatsMap);

        out = new CollectingInputFrameOut(BASICSCHEMA_INFO, queryNames, alqps, queries, mappingInfo, relnameStatsMap, prevFrameOut.outputLocation);

        backButton.setEnabled(true);
        nextButton.setEnabled(true);

        cos.write("Click Next to continue.");

    }

    private static Map<String, Statistics> fetchMetadata(Database clientDatabase, Set<String> relnames) {

//        Database clientDatabase;
//        try {
//            clientDatabase = new PostgresDatabase(database);
//        } catch (DatalessException ex) {
//            throw new RuntimeException("Unable to connect to Postgres Engine");
//        }

        Map<String, Statistics> relnameStatsMap = new HashMap<>();
        for (String relname : relnames) {
            try {
                relnameStatsMap.put(relname, TransferMode.fetchStatisics(relname, clientDatabase, null, "Saving", true));
            } catch (DatalessException ex) {
                throw new RuntimeException("Handle this");
            }
        }
        return relnameStatsMap;
    }

    private static Map<String, Statistics> loadMetadata(String folderLocation, Set<String> relnames) {

        Map<String, Statistics> relnameStatsMap = new HashMap<>();
        for (String relname : relnames) {
            String fileName = folderLocation + Constants.PathSeparator + relname;
            Constants.CPrintToConsole(" Reading statistics from file " + fileName, Constants.DEBUG_FIRST_LEVEL_Information);
            Statistics stats = null;
            try {
                FileInputStream fis = new FileInputStream(fileName);
                ObjectInputStream ois = new ObjectInputStream(fis);
                stats = (Statistics) ois.readObject();
                ois.close();
                relnameStatsMap.put(relname, stats);
                /**
                 * 1) Validate Relations in terms of the schema i.e. validate schema in source and destination are same
                 *      Current implementation assumes same schema is exists i.e same relation, attribute name and indexed columns in source and destination databases.
                 * 2) Provide option to map source relation or attributes to destination database, in case schema differs in naming
                 *      For example, the relation name could be different for source and destination database PART and PART1 and similarly for attributes.
                 */
            } catch (Exception ex) {
                throw new RuntimeException("Unable to deserialize relation statistics " + fileName);
            }
        }
        return relnameStatsMap;
    }

    /**
     * Collects metavalues appearing for each column in the metadata
     * Assuming columnnames are unique across tables
     * @param relnameStatsMap
     * @return
     */
    private HistogramMappingInfo getAsHistogramMappingInfo(Map<String, Statistics> relnameStatsMap) {

        exportToFile("del", new Gson().toJson(relnameStatsMap));

        HistogramMappingInfo mappingInfo = new HistogramMappingInfo();

        Set<String> numberTypes = new HashSet<>();
        numberTypes.add("int4");
        numberTypes.add("numeric");

        Set<String> stringTypes = new HashSet<>();
        stringTypes.add("bpchar");
        stringTypes.add("varchar");

        Set<String> dateTypes = new HashSet<>();
        dateTypes.add("date");
        dateTypes.add("time");

        for (Entry<String, Statistics> entry : relnameStatsMap.entrySet()) {
            String relname = entry.getKey();
            Statistics relstats = entry.getValue();

            for (Entry<String, ColumnStatistics> entry2 : relstats.getColumnStatistics().entrySet()) {
                String colname = entry2.getKey();
                PostgresColumnStatistics colstats = (PostgresColumnStatistics) entry2.getValue();

                String coltype = colstats.getColumnType();
                List<String> metavalues = new ArrayList<>();
                if (colstats.getHistogram() != null) {
                    for (HistogramObject obj : colstats.getHistogram().values()) {
                        metavalues.add(obj.getColValue());
                    }
                }
                if (colstats.getFrequencyHistogram() != null) {
                    for (DB2FreqHistObject obj : colstats.getFrequencyHistogram().values()) {
                        metavalues.add(obj.getColValue());
                    }
                }

                if (numberTypes.contains(coltype)) {
                    Map<Double, Integer> doubleMap = new HashMap<>();
                    mappingInfo.shownValuesByNumberColumns.put(colname, doubleMap);
                    for (String valstr : metavalues) {
                        doubleMap.put(Double.parseDouble(valstr), null);
                    }

                } else if (stringTypes.contains(coltype)) {
                    Map<String, Integer> stringMap = new HashMap<>();
                    mappingInfo.shownValuesByStringColumns.put(colname, stringMap);
                    for (String valstr : metavalues) {
                        stringMap.put(valstr, null);
                    }

                } else if (dateTypes.contains(coltype)) {
                    Map<Date, Integer> dateMap = new HashMap<>();
                    mappingInfo.shownValuesByDateColumns.put(colname, dateMap);
                    for (String valstr : metavalues) {
                        try {
                            dateMap.put(DATE_FORMAT.parse(valstr), null);
                        } catch (ParseException ex) {
                            throw new RuntimeException(ex);
                        }
                    }

                } else
                    throw new RuntimeException("Unrecognized coltype " + coltype);
            }
        }

        //exportToFile("del2", mappingInfo.toString());
        return mappingInfo;
    }

    private void exportToFile(String filename, String str) {
        try {
            FileWriter fw = new FileWriter(new File("/tmp/", filename));
            fw.write(str);
            fw.close();
        } catch (IOException ex) {
            throw new RuntimeException("File writing error", ex);
        }
    }

    private void onClickBack() {
        setVisible(false);
        dispose();
        cos.close();
        prevFrame.setVisible(true);
    }

    private void onClickNext() {
        setVisible(false);
        dispose();
        cos.close();
        new AnonymInputFrame(prevFrame, out).setVisible(true);
    }

}
