package iisc.dsl.coddgen.ui;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.DefaultTableModel;

import iisc.dsl.coddgen.ui.model.AnonymInputFrameOut;
import iisc.dsl.coddgen.ui.model.SolverFrameOut;
import iisc.dsl.coddgen.ui.utils.Constants;
import iisc.dsl.coddgen.ui.utils.Utils;
import in.ac.iisc.cds.dsl.cdgvendor.constants.PostgresVConfig;
import in.ac.iisc.cds.dsl.cdgvendor.model.CCInfo;
import in.ac.iisc.cds.dsl.cdgvendor.model.SolverViewStats;
import in.ac.iisc.cds.dsl.cdgvendor.model.ViewInfo;
import in.ac.iisc.cds.dsl.cdgvendor.model.ViewSolution;
import in.ac.iisc.cds.dsl.cdgvendor.model.ViewSolutionWithSolverStats;
import in.ac.iisc.cds.dsl.cdgvendor.model.formal.FormalCondition;
import in.ac.iisc.cds.dsl.cdgvendor.solver.DatabaseSummary;
import in.ac.iisc.cds.dsl.cdgvendor.solver.Z3Solver;
import in.ac.iisc.cds.dsl.cdgvendor.solver.Z3Solver.SolverType;
import in.ac.iisc.cds.dsl.cdgvendor.solver.Z3Solver.SpillType;
import in.ac.iisc.cds.dsl.cdgvendor.utils.DebugHelper;
import in.ac.iisc.cds.dsl.cdgvendor.utils.StopWatch;
import it.unimi.dsi.fastutil.doubles.Double2IntMap;
import it.unimi.dsi.fastutil.ints.Int2DoubleOpenHashMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.objects.Object2IntMap;

@SuppressWarnings("serial")
public class SolverFrame extends JFrame {

    private final JFrame              prevFrame;
    private final AnonymInputFrameOut prevFrameOut;

    private final JButton             backButton;
    private final JButton             staticDataButton;
    private final JButton             nextButton;

    private DefaultTableModel         tableModel;
    private Map<String, ViewSolution> uncompressedSummaryByView;
    private DatabaseSummary           databaseSummary;
    private final String			iLocation;
    private final String			oLocation;
    private final long				scaleFactor;

    public SolverFrame(JFrame prevFrame, AnonymInputFrameOut prevFrameOut, String iLocation, String oLocation, long scaleFactor) {

        this.prevFrame = prevFrame;
//        TODO : can remove prevFrameOut from picture
        this.prevFrameOut = prevFrameOut;
        
        this.iLocation = iLocation;
        this.oLocation = oLocation;
        this.scaleFactor = scaleFactor;

        backButton = new JButton("Back");
        staticDataButton = new JButton("Generate Static Data");
        nextButton = new JButton("Compare Query Plans");

        Utils.setFrameMedium(this, "Solver");
        initComponents();

        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    coddgenPostgres();
                } catch (Exception ex) {
                    Utils.showErrorDialog(SolverFrame.this, "Some Internal Error. Please check logs" + "\n" + ex.toString(), "Error");
                    return;
                }
                backButton.setEnabled(true);
                staticDataButton.setEnabled(true);
                nextButton.setEnabled(true);
            }
        }).start();
    }

    private void initComponents() {

        JPanel messagePanel = Utils.newSuccessBanner("Solving a constraint problem per relation");

        //headers for the table
        String[] columns = new String[] { "Status", "Relation", "# Var", "# Constr", "Solver Time", "# Rows", "Static Data?" };

        //actual data for the table in a 2d array
        Object[][] tableData = getEmptyObjectArr();

        final Class<Object>[] columnClass =
                new Class[] { Object.class, String.class, Integer.class, Integer.class, String.class, Integer.class, Boolean.class };

        //create table model with data
        tableModel = new DefaultTableModel(tableData, columns) {

            @Override
            public boolean isCellEditable(int row, int column) {
                return column == 6;
            }

            @Override
            public Class<?> getColumnClass(int columnIndex) {
                return columnClass[columnIndex];
            }
        };

        JTable table = new JTable(tableModel);
        table.setRowHeight(30);
        DefaultTableCellRenderer centerRenderer = new DefaultTableCellRenderer();
        centerRenderer.setHorizontalAlignment(JLabel.CENTER);
        table.setDefaultRenderer(String.class, centerRenderer);
        table.setDefaultRenderer(Integer.class, centerRenderer);
        table.getColumnModel().getColumn(0).setCellRenderer(new DefaultTableCellRenderer() {
            @Override
            public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int col) {
                if (value instanceof LoadingStatus) {
                    LoadingStatus loadingStatus = (LoadingStatus) value;
                    switch (loadingStatus) {
                        case QUEUED:
                            return new JLabel();
                        case SOLVING:
                            return new JLabel(Utils.createImageIcon("img/loading.png", "done"));
                        case SOLVED:
                            return new JLabel(Utils.createImageIcon("img/green_tick.png", "done"));
                        case FAILED:
                        default:
                            throw new RuntimeException("Unhandled case");
                    }
                }
                throw new RuntimeException("Unhandled case");
            }
        });
        //table.getColumnModel().getColumn(6).setCellRenderer(new JCheckBoxCellEditorRenderer());

        Action genStatic = new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                JTable table = (JTable) e.getSource();
                int modelRow = Integer.parseInt(e.getActionCommand());
                Boolean oldState = (Boolean) table.getModel().getValueAt(modelRow, 6);
                table.getModel().setValueAt(oldState ? Boolean.FALSE : Boolean.TRUE, modelRow, 6);
            }
        };
        CheckBoxColumn buttonColumn = new CheckBoxColumn(table, genStatic, 6);

        JPanel ctrlPanel = Utils.newCtlrsPanel();
        backButton.setEnabled(false);
        backButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                onClickBack();
            }
        });
        ctrlPanel.add(backButton);
        staticDataButton.setEnabled(false);
        staticDataButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                onClickStaticData();
            }
        });
        ctrlPanel.add(staticDataButton);
        nextButton.setEnabled(false);
        nextButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                onClickNext();
            }
        });
        if(prevFrameOut != null) {
        	Utils.setEnterButton(this, nextButton);
        	ctrlPanel.add(nextButton);
        }

        JPanel bgPanel = Utils.newBgPanel();
        bgPanel.setLayout(new BorderLayout());
        bgPanel.add(messagePanel, BorderLayout.NORTH);
        bgPanel.add(new JScrollPane(table), BorderLayout.CENTER);
        bgPanel.add(ctrlPanel, BorderLayout.SOUTH);
        getContentPane().add(bgPanel);
	}

    private void onClickBack() {
        setVisible(false);
        dispose();
        prevFrame.setVisible(true);
    }

    private void onClickStaticData() {

        List<String> viewnames = new ArrayList<>();
        for (int i = 0; i < tableModel.getRowCount(); i++) {
            if ((Boolean) tableModel.getValueAt(i, 6)) {
                viewnames.add(tableModel.getValueAt(i, 1).toString());
            }
        }
        if (viewnames.isEmpty()) {
            Utils.showErrorDialog(this, "Select atleast on relation for static generation", "Input Error");
            return;
        }

        JDialog processingDialog = new JDialog();
        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    StopWatch doubleStaticDumpSW = new StopWatch("DOUBLE-Static-Dump");
                    databaseSummary.dumpStaticRelations(viewnames);
                    doubleStaticDumpSW.displayTimeAndDispose();
                    processingDialog.dispose();
                    Utils.showInfoDialog(SolverFrame.this, "Done generating static data for " + viewnames.size() + " relations",
                            "Finished generating static data");
                } catch (Exception ex) {

                }
            }
        }).start();
        Utils.showProcessingDialog(processingDialog, this, "Generating static data for " + viewnames.size() + " relations", "Generating static data...");
    }

    private void onClickNext() {
        setVisible(false);
//        dispose();
        new CompareOutputFrame(this, new SolverFrameOut(prevFrameOut.inputFrameOut, prevFrameOut, uncompressedSummaryByView, scaleFactor), oLocation).setVisible(true);
    }

    public void coddgenPostgres() {

        SolverType solverType = SolverType.DOUBLE;
        SpillType spillType = SpillType.FILEBACKED_FKeyedBased;

        DebugHelper.printInfo("-------- CODDGEN: Started Postgres ------------");
        DebugHelper.printInfo("solverType: " + solverType + " spillType:" + spillType);

        StopWatch readInputSW = new StopWatch("Coddgen-ReadInputSW");
        //Step 1: Read SchemaInfo
        in.ac.iisc.cds.dsl.cdgvendor.constants.PostgresVConfig.loadAnonymizedViewInfos();

        //Step 1: Read CCInfo
        in.ac.iisc.cds.dsl.cdgvendor.constants.PostgresVConfig.loadCCInfo();
        CCInfo ccInfo = in.ac.iisc.cds.dsl.cdgvendor.constants.PostgresVConfig.CC_INFO;
        readInputSW.displayTimeAndDispose();

        //Step 2: Solver using z3
        Z3Solver solver = new Z3Solver(solverType, spillType);
        Map<String, ViewSolution> viewSolutions = solveEachView(solver, ccInfo);

        //Step 3: Create DatabaseSummary
        StopWatch databaseSummarySW = new StopWatch("Database-Summary");
        databaseSummary = new DatabaseSummary(databaseSummarySW, spillType, viewSolutions);
        databaseSummary.makeFKeyConsistency();
        //debugTotalErrorPerCondition(databaseSummarySW, databaseSummary, ccInfo);
        uncompressedSummaryByView = databaseSummary.getDuplicateUncompressedSummaryByView();
        databaseSummary.compressSummaryByAddingFkeys();
//        databaseSummary.dumpDatabaseSummary();
        
        Map<String, Double2IntMap> NUMBER_MAP = (Map<String, Double2IntMap>)Utils.readObjectFromFile(iLocation+Constants.PathSeparator+Constants.NUMBER_MAP_FILENAME);
        HashMap<String, Int2DoubleOpenHashMap> reverseNumberMap = new HashMap<>();
        for (Entry<String, Double2IntMap> entry : NUMBER_MAP.entrySet()) {
        	Int2DoubleOpenHashMap innerMap = new Int2DoubleOpenHashMap();
        	for (Entry<Double, Integer> innerEntry : entry.getValue().entrySet()) {
				innerMap.put(innerEntry.getValue().intValue(), innerEntry.getKey().doubleValue());
			}
        	reverseNumberMap.put(entry.getKey(), innerMap);
		}
        
        Map<String, Object2IntMap<String>> STRING_MAP = (Map<String, Object2IntMap<String>>)Utils.readObjectFromFile(iLocation+Constants.PathSeparator+Constants.STRING_MAP_FILENAME);
        HashMap<String, Int2ObjectOpenHashMap<String>> reverseStringMap = new HashMap<>();
        for (Entry<String, Object2IntMap<String>> entry : STRING_MAP.entrySet()) {
        	Int2ObjectOpenHashMap<String> innerMap = new Int2ObjectOpenHashMap<>();
        	for (Entry<String, Integer> innerEntry : entry.getValue().entrySet()) {
				innerMap.put(innerEntry.getValue().intValue(), innerEntry.getKey());
			}
        	reverseStringMap.put(entry.getKey(), innerMap);
		}
        
        Map<String, Object2IntMap<Date>> DATE_MAP = (Map<String, Object2IntMap<Date>>)Utils.readObjectFromFile(iLocation+Constants.PathSeparator+Constants.DATE_MAP_FILENAME);
        HashMap<String, Int2ObjectOpenHashMap<Date>> reverseDateMap = new HashMap<>();
        for (Entry<String, Object2IntMap<Date>> entry : DATE_MAP.entrySet()) {
        	Int2ObjectOpenHashMap<Date> innerMap = new Int2ObjectOpenHashMap<>();
        	for (Entry<Date, Integer> innerEntry : entry.getValue().entrySet()) {
				innerMap.put(innerEntry.getValue().intValue(), innerEntry.getKey());
			}
        	reverseDateMap.put(entry.getKey(), innerMap);
		}
        
        Map<String, String> TABLES_MAP = (Map<String, String>)Utils.readObjectFromFile(iLocation+Constants.PathSeparator+Constants.TABLE_MAP_FILENAME);
        HashMap<String, String> reverseTablesMap = new HashMap<>();
        for (Entry<String, String> entry : TABLES_MAP.entrySet()) {
        	reverseTablesMap.put(entry.getValue(), entry.getKey());
		}
        
        databaseSummary.decodeAndDumpDatabaseSummary(reverseNumberMap, reverseStringMap, reverseDateMap, reverseTablesMap);
        databaseSummarySW.displayTimeAndDispose();

        DebugHelper.printInfo("-------- CODDGEN: Finished Postgres ------------");
    }

    private Map<String, ViewSolution> solveEachView(Z3Solver solver, CCInfo ccInfo) {

        Map<String, ViewSolution> viewSolutions = new HashMap<>();

        Map<String, List<FormalCondition>> viewnameToCCMap = ccInfo.getViewnameToCCMap();
        List<String> sortedViewnames = new ArrayList<>(viewnameToCCMap.keySet());
        Collections.sort(sortedViewnames);

        updateUIWithAllViewsQueued(sortedViewnames);

        for (int i = 0; i < sortedViewnames.size(); i++) {
            String viewname = sortedViewnames.get(i);

            List<FormalCondition> conditions = viewnameToCCMap.get(viewname);
            ViewInfo viewInfo = PostgresVConfig.ANONYMIZED_VIEWINFOs.get(viewname);

            DebugHelper
                    .printInfo("\nSolving View: " + viewname + " (" + viewInfo.getViewNonkeys().size() + " dimensions | " + viewInfo.getRowcount() + " rows)");

            updateUIWithAViewSolving(i, viewname);

            ViewSolution viewSolution = solver.solveView(conditions, viewInfo, viewname, scaleFactor);

            if (!(viewSolution instanceof ViewSolutionWithSolverStats))
                throw new RuntimeException("Expected instanceof " + ViewSolutionWithSolverStats.class + " but got " + viewSolution.getClass());
            updateUIWithAViewSolved(i, viewname, ((ViewSolutionWithSolverStats) viewSolution).getSolverStats());

            //GCUtils.inviteGC();
            viewSolutions.put(viewname, viewSolution);
        }

        //Adding trivial solution for all view which didn't appear in viewnameToCCMap i.e., which had no constraints
        for (String viewname : PostgresVConfig.VIEWNAMES_TOPO) {
            if (!viewSolutions.keySet().contains(viewname)) {
                ViewInfo viewInfo = PostgresVConfig.ANONYMIZED_VIEWINFOs.get(viewname);
                ViewSolution viewSolution = solver.getTrivialSolution(viewInfo);
                viewSolutions.put(viewname, viewSolution);
            }
        }

        return viewSolutions;
    }

    // "Status", "Relation", "# Var", "# Constr", "Solver Time", "# Rows", "Static Data?"
    private static Object[][] getEmptyObjectArr() {
        Object[][] arr = new Object[0][7];
        return arr;
    }

    private void updateUIWithAllViewsQueued(List<String> viewnames) {
        for (String viewname : viewnames) {
            Object[] row = new Object[7];
            row[0] = LoadingStatus.QUEUED;
            row[1] = viewname;
            row[2] = null;
            row[3] = null;
            row[4] = null;
            row[5] = null;
            row[6] = null;
            tableModel.addRow(row);
        }
    }

    private void updateUIWithAViewSolving(int rowNum, String viewname) {
        Object[] row = new Object[7];
        row[0] = LoadingStatus.SOLVING;
        row[1] = viewname;
        row[2] = null;
        row[3] = null;
        row[4] = null;
        row[5] = null;
        row[6] = null;
        tableModel.removeRow(rowNum);
        tableModel.insertRow(rowNum, row);
    }

    private void updateUIWithAViewSolved(int rowNum, String viewname, SolverViewStats solverStats) {
        Object[] row = new Object[7];
        row[0] = LoadingStatus.SOLVED;
        row[1] = viewname;
        row[2] = solverStats.solverVarCount;
        row[3] = solverStats.solverConstraintCount;
        row[4] = solverStats.millisToSolve < 1000 ? "<1s" : String.format("%.1fs", solverStats.millisToSolve / 1000.0);
        row[5] = solverStats.relRowCount;
        row[6] = Boolean.FALSE;
        tableModel.removeRow(rowNum);
        tableModel.insertRow(rowNum, row);
    }

    private enum LoadingStatus {
        QUEUED,
        SOLVING,
        SOLVED,
        FAILED;
    }

}
