package in.ac.iisc.cds.dsl.cdg.main;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import in.ac.iisc.cds.dsl.cdgclient.anonymizer.Anonymizer;
import in.ac.iisc.cds.dsl.cdgclient.anonymizer.SchemaAnonymize;
import in.ac.iisc.cds.dsl.cdgclient.anonymizer.SchemaAnonymizePostgres;
import in.ac.iisc.cds.dsl.cdgclient.constants.PostgresCConfig;
import in.ac.iisc.cds.dsl.cdgclient.constants.PostgresCConfig.Key;
import in.ac.iisc.cds.dsl.cdgclient.preprocess.AlqpToCC;
import in.ac.iisc.cds.dsl.cdgclient.preprocess.AlqpToCCPostgres;
import in.ac.iisc.cds.dsl.cdgclient.preprocess.DDLGenerator;
import in.ac.iisc.cds.dsl.cdgclient.preprocess.ExplainAnalyzeToAlqp;
import in.ac.iisc.cds.dsl.cdgclient.preprocess.ExplainAnalyzeToAlqpPostgres;
import in.ac.iisc.cds.dsl.cdgclient.preprocess.QueryToExplainAnalyze;
import in.ac.iisc.cds.dsl.cdgclient.preprocess.QueryToExplainAnalyzePostgres;
import in.ac.iisc.cds.dsl.cdgclient.preprocess.SchemaAnalyze;
import in.ac.iisc.cds.dsl.cdgclient.preprocess.SchemaAnalyzePostgres;
import in.ac.iisc.cds.dsl.cdgvendor.constants.PostgresVConfig;
import in.ac.iisc.cds.dsl.cdgvendor.model.Alqp;
import in.ac.iisc.cds.dsl.cdgvendor.model.CCInfo;
import in.ac.iisc.cds.dsl.cdgvendor.model.HistogramMappingInfo;
import in.ac.iisc.cds.dsl.cdgvendor.model.ViewInfo;
import in.ac.iisc.cds.dsl.cdgvendor.model.ViewSolution;
import in.ac.iisc.cds.dsl.cdgvendor.model.formal.FormalCondition;
import in.ac.iisc.cds.dsl.cdgvendor.solver.DatabaseSummary;
import in.ac.iisc.cds.dsl.cdgvendor.solver.Z3Solver;
import in.ac.iisc.cds.dsl.cdgvendor.solver.Z3Solver.SolverType;
import in.ac.iisc.cds.dsl.cdgvendor.solver.Z3Solver.SpillType;
import in.ac.iisc.cds.dsl.cdgvendor.utils.ConditionsEvaluator;
import in.ac.iisc.cds.dsl.cdgvendor.utils.DebugHelper;
import in.ac.iisc.cds.dsl.cdgvendor.utils.FileUtils;
import in.ac.iisc.cds.dsl.cdgvendor.utils.StopWatch;

public class Main {

    private static void checkHeapSize() {
        // Get current size of heap in bytes
        long heapSize = Runtime.getRuntime().totalMemory();

        // Get maximum size of heap in bytes. The heap cannot grow beyond this size.// Any attempt will result in an OutOfMemoryException.
        long heapMaxSize = Runtime.getRuntime().maxMemory();

        // Get amount of free memory within the heap in bytes. This size will increase // after garbage collection and decrease as new objects are created.
        long heapFreeSize = Runtime.getRuntime().freeMemory();

        System.out.println(heapSize / (1024 * 1024));
        System.out.println(heapMaxSize / (1024 * 1024));
        System.out.println(heapFreeSize / (1024 * 1024));
    }

    public static void main(String[] args) {

//        checkHeapSize();
    	
        Map<Key, String> overlapConfig = new HashMap<>();
        
        /************************************************************
         * This call produces explain-analyze from given queries.
         * Takes a lot of time. Keep it commented until you have
         * changed the queries
         ************************************************************/
        PostgresCConfig.initDefaultConfig();
        PostgresCConfig.overlayOnDefaultConfig(overlapConfig);
        
        generateEAsPostgres();

        /************************************************************
         * This call produces cardinality constraints from given EAs.
         * EAs are traversed to extract join and filter conditions.
         * Then tablenames and columnnames are anonymized.
         * Also all literal values appearing in the constraints (other
         * than output cardinalities) are mapped to integer domains
         ************************************************************/
        generateAnonymizedCCsPostgres();

        /************************************************************
         * This call triggers the data generation module which
         * takes anonymized CCs as input and produces database summary
         ************************************************************/
        PostgresVConfig.initDefaultConfig();
        coddgenPostgres();
    }

    private static void generateEAsPostgres() {
        DebugHelper.printInfo("-------- Gateway: Generating EAs for Postgres ------------");

//        Step 1: Fetch the database schema
        SchemaAnalyze sa = new SchemaAnalyzePostgres();
        sa.explainSchema();

//        Step 2: Fetch explain-analyze for all given queries
		List<String> queryIndex = FileUtils.readLines(PostgresCConfig.getProp(Key.SQLQUERIES_LOCATION), PostgresCConfig.SQLQUERIES_INDEX);
		List<String> queriesFromFiles = new ArrayList<>();
		List<String> queryNames = new ArrayList<>();
		for (String queryFilename : queryIndex) {
			String sqlQuery = FileUtils.readFileToString(PostgresCConfig.getProp(Key.SQLQUERIES_LOCATION), queryFilename);
			queriesFromFiles.add(sqlQuery);
			queryNames.add(queryFilename);
		}
		QueryToExplainAnalyze ea = new QueryToExplainAnalyzePostgres();
		ea.explainAnalyzeQueries(queriesFromFiles, queryNames);

        DebugHelper.printInfo("-------- Gateway: Done generating EAs for Postgres ------------");
    }

    private static void generateAnonymizedCCsPostgres() {
        DebugHelper.printInfo("-------- Gateway: Generating anonymized CCs for Postgres ------------");

        //Step 1: Read schema
        PostgresCConfig.loadBasicSchemaInfo();

        //Step 2: Anonymize it while creating anonymizer
        SchemaAnonymize san = new SchemaAnonymizePostgres();
        Anonymizer anonymizer = san.anonymizeSchema();

        //Step 3: Generate ddl for anonymized schema. Also contains primary key/foreign key constraints
//        DDLGenerator dg = new DDLGenerator();
//        dg.generateDDL(anonymizer);

        //Step 4: Read all given explain-analyze as ALQPs
        List<String> eaIndex = FileUtils.readLines(PostgresCConfig.getProp(Key.EXPANALYZE_LOCATION), PostgresCConfig.EXPANALYZE_INDEX);
        List<String> easFromFiles = new ArrayList<>();	// EA, Plan in Json String
        List<String> queryNames = new ArrayList<>();
        for (String eaFilename : eaIndex) {
            String eaStr = FileUtils.readFileToString(PostgresCConfig.getProp(Key.EXPANALYZE_LOCATION), eaFilename);
            easFromFiles.add(eaStr);
            queryNames.add(eaFilename);
        }
        ExplainAnalyzeToAlqp ea = new ExplainAnalyzeToAlqpPostgres();
        List<Alqp> alqps = ea.createAllAlqp(easFromFiles, queryNames);

        //Step 5: Generate anonymized CCs and queries with domain mapping
        HistogramMappingInfo emptyHistogramMappingInfo = new HistogramMappingInfo();
        AlqpToCC ac = new AlqpToCCPostgres(anonymizer);
        ac.anonymizeAlqpsAndGenerateCCsAndQueries(alqps, emptyHistogramMappingInfo);

        DebugHelper.printInfo("-------- Gateway: Done generating anonymized CCs for Postgres ------------");
    }

    private static void coddgenPostgres() {

        SolverType solverType = SolverType.DOUBLE;
        SpillType spillType = SpillType.INMEMORY;

        DebugHelper.printInfo("-------- CODDGEN: Started Postgres ------------");
        DebugHelper.printInfo("solverType: " + solverType + " spillType:" + spillType);

        StopWatch readInputSW = new StopWatch("Coddgen-ReadInputSW");
        //Step 1: Read SchemaInfo
        in.ac.iisc.cds.dsl.cdgvendor.constants.PostgresVConfig.loadAnonymizedViewInfos();

        //Step 1: Read CCInfo
        in.ac.iisc.cds.dsl.cdgvendor.constants.PostgresVConfig.loadCCInfo();
        CCInfo ccInfo = in.ac.iisc.cds.dsl.cdgvendor.constants.PostgresVConfig.CC_INFO;
        readInputSW.displayTimeAndDispose();

        //Step 2: Solver using z3
        Z3Solver solver = new Z3Solver(solverType, spillType);
        Map<String, ViewSolution> viewSolutions = solveEachView(solver, ccInfo);

        //Step 3: Create DatabaseSummary
        StopWatch databaseSummarySW = new StopWatch("Database-Summary");
        DatabaseSummary databaseSummary = new DatabaseSummary(databaseSummarySW, spillType, viewSolutions);
        databaseSummary.makeFKeyConsistency();
        debugTotalErrorPerCondition(databaseSummarySW, databaseSummary, ccInfo);
        databaseSummary.compressSummaryByAddingFkeys();
        if (solverType == SolverType.DOUBLE) {
            databaseSummary.dumpDatabaseSummary();
        }
        databaseSummarySW.displayTimeAndDispose();
        
//        if (solverType == SolverType.DOUBLE) {
//            StopWatch doubleStaticDumpSW = new StopWatch("DOUBLE-Static-Dump");
//            databaseSummary.dumpAllStaticRelations();
//            doubleStaticDumpSW.displayTimeAndDispose();
//        }

        DebugHelper.printInfo("-------- CODDGEN: Finished Postgres ------------");
    }

    private static Map<String, ViewSolution> solveEachView(Z3Solver solver, CCInfo ccInfo) {

        Map<String, ViewSolution> viewSolutions = new HashMap<>();

        Map<String, List<FormalCondition>> viewnameToCCMap = ccInfo.getViewnameToCCMap();
        List<String> sortedViewnames = new ArrayList<>(viewnameToCCMap.keySet());
        Collections.sort(sortedViewnames);
        for (String viewname : sortedViewnames) {
            List<FormalCondition> conditions = viewnameToCCMap.get(viewname);
            ViewInfo viewInfo = PostgresVConfig.ANONYMIZED_VIEWINFOs.get(viewname);

            DebugHelper
                    .printInfo("\nSolving View: " + viewname + " (" + viewInfo.getViewNonkeys().size() + " dimensions | " + viewInfo.getRowcount() + " rows)");

            ViewSolution viewSolution = solver.solveView(conditions, viewInfo, viewname, 1);
            //GCUtils.inviteGC();
            viewSolutions.put(viewname, viewSolution);
        }

        //Adding trivial solution for all view which didn't appear in viewnameToCCMap i.e., which had no constraints
        for (String viewname : PostgresVConfig.VIEWNAMES_TOPO) {
            if (!viewSolutions.keySet().contains(viewname)) {
                ViewInfo viewInfo = PostgresVConfig.ANONYMIZED_VIEWINFOs.get(viewname);
                ViewSolution viewSolution = solver.getTrivialSolution(viewInfo);
                viewSolutions.put(viewname, viewSolution);
            }
        }

        return viewSolutions;
    }

    private static void debugTotalErrorPerCondition(StopWatch databaseSummarySW, DatabaseSummary databaseSummary, CCInfo ccInfo) {
        if (!DebugHelper.totalErrorCheckNeeded())
            return;

        databaseSummarySW.pause();
        DebugHelper.printDebug("Evaluating total errors per condition");
        List<String> sortedViewnames = new ArrayList<>(databaseSummary.getUncompressedSummaryByView().keySet());
        Collections.sort(sortedViewnames);
        for (String viewname : sortedViewnames) {
            ViewInfo viewInfo = PostgresVConfig.ANONYMIZED_VIEWINFOs.get(viewname);
            List<String> sortedColumns = new ArrayList<>(viewInfo.getViewNonkeys());
            Collections.sort(sortedColumns);

            if (ccInfo.getViewnameToCCMap().containsKey(viewname)) {
                DebugHelper.printDebug("Evaluating total errors per condition for view " + viewname);
                ConditionsEvaluator.debugErrorPerCondition(ccInfo.getViewnameToCCMap().get(viewname),
                        databaseSummary.getUncompressedSummaryByView().get(viewname), sortedColumns);
            }
        }
        databaseSummarySW.resume();
    }

}
