package in.ac.iisc.cds.dsl.cdgclient.preprocess;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import in.ac.iisc.cds.dsl.cdgclient.anonymizer.Anonymizer;
import in.ac.iisc.cds.dsl.cdgclient.constants.PostgresCConfig;
import in.ac.iisc.cds.dsl.cdgclient.constants.PostgresCConfig.Key;
import in.ac.iisc.cds.dsl.cdgvendor.model.Alqp;
import in.ac.iisc.cds.dsl.cdgvendor.model.CCInfo;
import in.ac.iisc.cds.dsl.cdgvendor.model.Condition;
import in.ac.iisc.cds.dsl.cdgvendor.model.HistogramMappingInfo;
import in.ac.iisc.cds.dsl.cdgvendor.model.formal.FormalCondition;
import in.ac.iisc.cds.dsl.cdgvendor.utils.Converter;
import in.ac.iisc.cds.dsl.cdgvendor.utils.DebugHelper;
import in.ac.iisc.cds.dsl.cdgvendor.utils.SerializationHelper;
//import jdk.internal.org.objectweb.asm.tree.AnnotationNode;

/**
 * Reads explain analyse jsons and extracts conditions out of it.
 * Also anonymizes the constraints
 * @author raghav
 *
 */
public class AlqpToCCPostgres extends AlqpToCC {

    private static final String NEWLINE = "\n";

    public AlqpToCCPostgres(Anonymizer anonymizer) {
        super(anonymizer);
    }

    @Override
    public AlqpToCCPostgresRes anonymizeAlqpsAndGenerateCCsAndQueries(List<Alqp> alqps, HistogramMappingInfo histogramMappingInfo) {

        DebugHelper.printInfo("-------- Generating anonymized CCs from " + alqps.size() + " Postgres ALQPs ------------");
        
        System.err.println(anonymizer.getTablenameAnonymMap());
        System.err.println(anonymizer.getColumnnameAnonymMap());

        List<List<FormalCondition>> alqpFormalConditions = new ArrayList<>(alqps.size());

        List<FormalCondition> allFormalConditions = new ArrayList<>();
        for (int i = 0; i < alqps.size(); i++) {
            Alqp alqp = alqps.get(i);
            DebugHelper.printDebug("\nAlqp " + (i + 1) + "/" + alqps.size() + ": ");

            //Step 1: Extract CCs
            List<Condition> someConditions = alqp.getAllConditions();
            DebugHelper.printDebug("\nAlqp conditions: ");
            DebugHelper.printConditions(someConditions);

            List<FormalCondition> someFormalConditions = Converter.getAsFormalConditions(someConditions, anonymizer);
            DebugHelper.printDebug("\nAlqp formal conditions: ");
            DebugHelper.printFormalConditions(someFormalConditions);

            alqpFormalConditions.add(deepCopy(someFormalConditions));
            allFormalConditions.addAll(someFormalConditions);
        }

        //Step 2: Anonymize CCs
        anonymizer.anonymizeAllFormalConditions(allFormalConditions);
        DebugHelper.printDebug("\nAll anonym formal conditions: ");
        DebugHelper.printFormalConditions(allFormalConditions);

        List<String> anonymQueries = new ArrayList<>();
        //Step 3: Domainmap CCs
        List<FormalCondition> mappedFormalConditions =
                anonymizer.domainmapAllFormalConditionsAndGenerateQueries(allFormalConditions, histogramMappingInfo, alqps, anonymQueries);

        List<List<FormalCondition>> alqpAnonymFormalConditions = new ArrayList<>(alqps.size());
        for (int i = 0, lastSeenIndx = -1; i < alqps.size(); i++) {
            int startIndx = lastSeenIndx + 1; //included
            int endIndx = startIndx + alqpFormalConditions.get(i).size(); //excluded
            List<FormalCondition> someAnonymFormalConditions = mappedFormalConditions.subList(startIndx, endIndx);
            alqpAnonymFormalConditions.add(someAnonymFormalConditions);
            lastSeenIndx = endIndx - 1;
        }

        //Step 4: Unique Domainmap CCs
        List<FormalCondition> uniqueMappedFormalConditions = new ArrayList<>(new HashSet<>(mappedFormalConditions));

        DebugHelper.printDebug("All unique domainmapped anonym formal conditions (" + mappedFormalConditions.size() + " uniqued "
                + uniqueMappedFormalConditions.size() + "): ");
        DebugHelper.printFormalConditions(uniqueMappedFormalConditions);

        //Step 5: Dump CCs
        Map<String, List<FormalCondition>> viewnameToCCMap = new HashMap<>();
        for (FormalCondition mappedFormalCondition : uniqueMappedFormalConditions) {
            String viewname = mappedFormalCondition.getViewname();
            List<FormalCondition> ccList = viewnameToCCMap.get(viewname);
            if (ccList == null) {
                ccList = new ArrayList<>();
                viewnameToCCMap.put(viewname, ccList);
            }
            ccList.add(mappedFormalCondition);
        }

        CCInfo ccInfo = new CCInfo();
        ccInfo.setViewnameToCCMap(viewnameToCCMap);
        SerializationHelper.serializeCCInfo(ccInfo, PostgresCConfig.getProp(Key.ANONYMIZEDCCS_TARGETFILE));
        DebugHelper.printInfo("-------- Done Generating anonymized CCs from " + alqps.size() + " Postgres ALQPs ------------");

        AlqpToCCPostgresRes res = new AlqpToCCPostgresRes(alqpFormalConditions, alqpAnonymFormalConditions, anonymQueries);
        return res;
    }

    //    private static List<Condition> getAllConditions(List<Alqp> alqps) {
    //
    //        List<Condition> allconditions = new ArrayList<>();
    //        for (Alqp alqp : alqps) {
    //            List<Condition> conditions = alqp.getAllConditions();
    //            allconditions.addAll(conditions);
    //        }
    //        return allconditions;
    //
    //    }

    private List<FormalCondition> deepCopy(List<FormalCondition> formalConditions) {
        List<FormalCondition> another = new ArrayList<>();
        for (FormalCondition formalCondition : formalConditions) {
            another.add(formalCondition.getDeepCopy());
        }
        return another;
    }
}
