package in.ac.iisc.cds.dsl.cdgvendor.constants;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.json.JSONObject;

import in.ac.iisc.cds.dsl.cdgvendor.model.CCInfo;
import in.ac.iisc.cds.dsl.cdgvendor.model.SchemaInfo;
import in.ac.iisc.cds.dsl.cdgvendor.model.TableInfo;
import in.ac.iisc.cds.dsl.cdgvendor.model.ViewInfo;
import in.ac.iisc.cds.dsl.cdgvendor.utils.ConfigProvider;
import in.ac.iisc.cds.dsl.cdgvendor.utils.FileUtils;
import in.ac.iisc.cds.dsl.cdgvendor.utils.SerializationHelper;

/**
 * PostgresVendorConfig
 * @author dsladmin
 *
 */
public class PostgresVConfig {

    public static final Charset     CHARSET               = Charset.forName("UTF-8");

    private static final String     DEFAULTPROPS_FILENAME = "cdgvendor/postgres.properties";
    private static Map<Key, String> PROPS_MAP             = null;

    public static void initDefaultConfig() {
        PROPS_MAP = new HashMap<>();
        Map<String, String> temp = ConfigProvider.INSTANCE.getPropsFromFile(PostgresVConfig.DEFAULTPROPS_FILENAME);
        for (Key key : Key.values()) {
            PROPS_MAP.put(key, temp.get(key.configFileKey));
        }
    }

    public static void overlayOnDefaultConfig(Map<Key, String> overlayConfig) {
        for (Entry<Key, String> entry : overlayConfig.entrySet()) {
            PROPS_MAP.put(entry.getKey(), entry.getValue());
        }
    }

    public enum Key {

        ANONYMIZEDCCS_INFO("postgres.anonymizedcc.info"),
        ANONYMIZEDSCHEMA_FILENAME("postgres.anonymizedschema.info"),

        DATABASESUMMARY_LOCATION("postgres.databasesummary.location"),
        DATABASESTATICDUMP_LOCATION("postgres.databasesummary.staticdumplocation");

        String configFileKey;

        private Key(String configFileKey) {
            this.configFileKey = configFileKey;
        }
    }

    public static String getProp(Key key) {
        return PROPS_MAP.get(key);
    }

    public static final String DATABASESUMMARY_INDEX = "index";

    public static CCInfo       CC_INFO               = null;

    public static void loadCCInfo() {
        if (CC_INFO == null) {
            CC_INFO = SerializationHelper.deserializeCCInfo(getProp(Key.ANONYMIZEDCCS_INFO));
        }
    }

    public static Map<String, ViewInfo> ANONYMIZED_VIEWINFOs = null;
    public static List<String>          VIEWNAMES_TOPO       = null;
    public static List<String>          VIEWNAMES_REVTOPO    = null;
    public static SchemaInfo			ANONYMIZED_SCHEMAINFO = null;

    public static void loadAnonymizedViewInfos() {
        if (ANONYMIZED_VIEWINFOs == null) {
            try {
                JSONObject obj = new JSONObject(FileUtils.readFileToString(getProp(Key.ANONYMIZEDSCHEMA_FILENAME)));
                ANONYMIZED_SCHEMAINFO = new SchemaInfo();
                for (String key : obj.keySet()) {
                	ANONYMIZED_SCHEMAINFO.putTableInfo(key, new TableInfo(obj.getJSONObject(key)));
                }
                ANONYMIZED_SCHEMAINFO.validate();

                //TODO: Use some ViewInfo and ViewGen instead of SchemaInfo and TableInfo
                //Populate nonKeys attributes bottom-up
                VIEWNAMES_REVTOPO = getRevTopoList(ANONYMIZED_SCHEMAINFO.getTableInfos());
                VIEWNAMES_TOPO = new ArrayList<>(VIEWNAMES_REVTOPO);
                Collections.reverse(VIEWNAMES_TOPO);

                int k = -1;
                ANONYMIZED_VIEWINFOs = new LinkedHashMap<>();
                for (String tablename : VIEWNAMES_REVTOPO) {
                    TableInfo tableInfo = ANONYMIZED_SCHEMAINFO.getTableInfo(tablename);

                    Set<String> tableNonkeys = new HashSet<>();
                    tableNonkeys.addAll(tableInfo.columnsNames());
                    tableNonkeys.removeAll(tableInfo.getKeys());

                    Set<String> viewNonkeys = new HashSet<>();
                    viewNonkeys.addAll(tableNonkeys);
                    for (String immediateTarget : tableInfo.getFkeyTables()) {
                        viewNonkeys.addAll(ANONYMIZED_VIEWINFOs.get(immediateTarget).getViewNonkeys());
                    }
                    ViewInfo viewInfo = new ViewInfo(++k, tableInfo.getRowcount(), tableInfo.getFkeyTables());
                    viewInfo.getTableNonkeys().addAll(tableNonkeys);
                    viewInfo.getViewNonkeys().addAll(viewNonkeys);
                    ANONYMIZED_VIEWINFOs.put(tablename, viewInfo);
                }

                //Setting isNeverFKeyed field in viewInfo
                for (String tablename : VIEWNAMES_REVTOPO) {
                    ViewInfo viewInfo = ANONYMIZED_VIEWINFOs.get(tablename);
                    viewInfo.setIsNeverFKeyed(true);
                }
                for (String tablename : VIEWNAMES_REVTOPO) {
                    ViewInfo viewInfo = ANONYMIZED_VIEWINFOs.get(tablename);
                    for (String fkeyedViewname : viewInfo.getFkeyViews()) {
                        ViewInfo fkeyedViewInfo = ANONYMIZED_VIEWINFOs.get(fkeyedViewname);
                        fkeyedViewInfo.setIsNeverFKeyed(false);
                    }
                }
            } catch (Exception ex) {
                throw new ExceptionInInitializerError(ex);
            }
        }
    }

    private static List<String> getRevTopoList(Map<String, TableInfo> tableInfos) {

        Map<String, Boolean> done = new HashMap<>();

        for (Entry<String, TableInfo> entry : tableInfos.entrySet()) {
            String tablename = entry.getKey();
            done.put(tablename, false);
        }

        List<String> revTopoList = new ArrayList<>();
        int max;
        while (true) {
            max = Integer.MIN_VALUE;
            String toAddTablename = null;
            for (Entry<String, TableInfo> entry : tableInfos.entrySet()) {
                String tablename = entry.getKey();
                TableInfo tableInfo = entry.getValue();

                if (!done.get(tablename) && tableInfo.getTopoSeqno() > max) {
                    max = tableInfo.getTopoSeqno();
                    toAddTablename = tablename;
                }
            }
            if (max == Integer.MIN_VALUE) {
                break;
            }
            revTopoList.add(toAddTablename);
            done.put(toAddTablename, true);
        }
        return revTopoList;
    }

}
