package in.ac.iisc.cds.dsl.cdgvendor.solver;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import in.ac.iisc.cds.dsl.cdgvendor.model.ValueCombination;
import in.ac.iisc.cds.dsl.cdgvendor.model.ViewInfo;
import in.ac.iisc.cds.dsl.cdgvendor.model.ViewSolution;
import in.ac.iisc.cds.dsl.cdgvendor.model.ViewSolutionInMemory;
import in.ac.iisc.cds.dsl.cdgvendor.model.formal.FormalCondition;
import in.ac.iisc.cds.dsl.cdgvendor.model.formal.FormalConditionAnd;
import in.ac.iisc.cds.dsl.cdgvendor.model.formal.FormalConditionOr;
import in.ac.iisc.cds.dsl.cdgvendor.model.formal.FormalConditionSOP;
import in.ac.iisc.cds.dsl.cdgvendor.model.formal.FormalConditionSimpleInt;
import in.ac.iisc.cds.dsl.cdgvendor.model.formal.Symbol;
import in.ac.iisc.cds.dsl.cdgvendor.reducer.Bucket;
import in.ac.iisc.cds.dsl.cdgvendor.reducer.BucketStructure;
import in.ac.iisc.cds.dsl.cdgvendor.reducer.Region;
import in.ac.iisc.cds.dsl.cdgvendor.utils.ConditionsEvaluator;
import in.ac.iisc.cds.dsl.cdgvendor.utils.DebugHelper;
import in.ac.iisc.cds.dsl.cdgvendor.utils.StopWatch;
import it.unimi.dsi.fastutil.ints.IntArrayList;
import it.unimi.dsi.fastutil.ints.IntList;

public class Z3Solver {

    private static final DecimalFormat DF = new DecimalFormat("0", DecimalFormatSymbols.getInstance(Locale.ENGLISH));
    static {
        DF.setMaximumFractionDigits(340);
    }

    public enum SolverType {
        ARASU,
        DOUBLE;
    }

    public enum SpillType {
        INMEMORY,
        //        MAPDBBACKED,
        //        FILEBACKED,
        FILEBACKED_FKeyedBased;
    }

    private final SolverType solverType;
    /**
     * Applicable in case of ARASU
     */
    private final SpillType  spillType;

    public Z3Solver(SolverType solverType, SpillType spillType) {
        this.solverType = solverType;
        this.spillType = spillType;
    }

    public ViewSolution getTrivialSolution(ViewInfo viewInfo) {
        IntList colValues = new IntArrayList(viewInfo.getViewNonkeys().size());
        for (int i = viewInfo.getViewNonkeys().size() - 1; i >= 0; i--) {
            colValues.add(0);
        }
        ViewSolutionInMemory viewSolution = new ViewSolutionInMemory(1);
        viewSolution.addValueCombination(new ValueCombination(colValues, viewInfo.getRowcount()));
        return viewSolution;
    }
    
    public ViewSolution solveView(List<FormalCondition> conditions, ViewInfo viewInfo, String viewname, long scaleFactor) {
        StopWatch preprocessViewSW = new StopWatch("Pre-Processing-" + viewname);
        Map<String, IntList> bucketFloorsByColumns = DomainDecomposer.getBucketFloors(conditions, viewInfo);

        final List<String> sortedViewColumns = new ArrayList<>(bucketFloorsByColumns.keySet());
        Collections.sort(sortedViewColumns);

        //Marking buckets for each column as true
        double varcountUR = 1;
        final List<boolean[]> allTrueBS = new ArrayList<>();
        for (String column : sortedViewColumns) {
            int length = bucketFloorsByColumns.get(column).size();
            boolean[] arr = new boolean[length];
            varcountUR *= length;

            Arrays.fill(arr, true);
            allTrueBS.add(arr);
        }

        DebugHelper.printInfo("Number of variables without reduction is " + DF.format(varcountUR));

        if (solverType == SolverType.DOUBLE) {
            //NOTE: Scaling only to extent when cardinality value stays within permissible range of long
            //NOTE: Scaling such that rowcount of any PK table goes beyond range of int will introduce negative values in DatabaseSummary.
            //TODO : This is not yet fixed.

            //long scaleFactor = (long) 1e10;
            DebugHelper.printInfo("SCALING by " + scaleFactor);
            viewInfo.scaleRowCount(scaleFactor);
            for (FormalCondition condition : conditions) {
                condition.scaleRowCount(scaleFactor);
            }
        }

        CliqueFinder cliqueReducer = new CliqueFinder(viewInfo, allTrueBS);
        List<Set<String>> arasuCliques = cliqueReducer.getOrderedNonTrivialCliques(conditions);
  
///////////////// Start dk
//        Disable arasu clique optimisation
        /*List<String> sortedColumns = new ArrayList<>(viewInfo.getViewNonkeys());
        Set<String> temp = new HashSet<>(sortedColumns);
        arasuCliques.clear();
        arasuCliques.add(temp);*/
///////////////// End dk
        
        long varcountAR = cliqueReducer.getReducedVariableCount(arasuCliques);
        DebugHelper.printInfo("Number of variables using Arasu's clique based reduction " + varcountAR);

        if (varcountAR > varcountUR)
            throw new RuntimeException("Arasu's reduction is increasing variables over unreduced");

///////////////// Start dk
        StopWatch getProjectedConditionsSW = new StopWatch("getProjectedConditions");
        List<List<FormalCondition>> conditionsInSOPForm = new ArrayList<>();
        for (FormalCondition condition : conditions) {
        	List<FormalCondition> conditionInSOPForm = getSOPSubconditions(condition);	// Note : outputCardinality is lost
        	conditionsInSOPForm.add(conditionInSOPForm);
		}
        
        HashMap<FormalConditionOr, Integer> consistencyConstraintsToIndex = new HashMap<>();
        int newCCIndex = 0;
        
        Set<Set<String>> processedCommonAttribs = new HashSet<>();
        for (int i = 0; i < arasuCliques.size(); i++) {
        	Set<String> c1 = arasuCliques.get(i);
        	for (int j = i + 1; j < arasuCliques.size(); j++) {
        		Set<String> c2 = arasuCliques.get(j);
        		Set<String> commonAttribsSet = new HashSet<>(c1);
        		commonAttribsSet.retainAll(c2);
        		if(!commonAttribsSet.isEmpty()) {
        			if(processedCommonAttribs.contains(commonAttribsSet))
        				continue;
        			else
        				processedCommonAttribs.add(commonAttribsSet);
        			Set<FormalConditionOr> newConsistencyConstraints = getProjectedConditions(commonAttribsSet, conditionsInSOPForm, allTrueBS, bucketFloorsByColumns, sortedViewColumns);
        			for (FormalConditionOr formalConditionOr : newConsistencyConstraints) {
        				if(!consistencyConstraintsToIndex.containsKey(formalConditionOr)) {
        					consistencyConstraintsToIndex.put(formalConditionOr, newCCIndex);
        					newCCIndex++;
        				}
					}
        		}
			}
		}
//        Converting to list to maintain ordering
        FormalConditionOr consistencyConstraints[] = new FormalConditionOr[consistencyConstraintsToIndex.size()];
        for (Entry<FormalConditionOr, Integer> entry : consistencyConstraintsToIndex.entrySet()) {
			consistencyConstraints[entry.getValue()] = entry.getKey();
		}
        
        for (FormalConditionOr formalCondition : consistencyConstraints) {
        	conditionsInSOPForm.add(formalCondition.getConditionList());
		}

        getProjectedConditionsSW.displayTimeAndDispose();
///////////////// End dk
        List<Region> conditionRegions = getConditionRegions(conditionsInSOPForm, allTrueBS, sortedViewColumns, bucketFloorsByColumns);
        
        DebugHelper.printInfo("Number of cardinality constraints " + (conditionRegions.size() - consistencyConstraints.length + 1)); //All 1's which is added later is included in the count here

        ViewSolution fullViewSolution = null;
        switch (solverType) {
            case ARASU:
                ArasuReductionBasedViewSolver arasuSolver =
                        new ArasuReductionBasedViewSolver(viewname, viewInfo, allTrueBS, arasuCliques, bucketFloorsByColumns, spillType);
                preprocessViewSW.displayTimeAndDispose();
                fullViewSolution = arasuSolver.solveView(conditions, new ArrayList<>(conditionRegions), null);
                break;

            case DOUBLE:
                DoubleReductionBasedViewSolver doubleSolver =
                        new DoubleReductionBasedViewSolver(viewname, viewInfo, allTrueBS, arasuCliques, bucketFloorsByColumns);
                preprocessViewSW.displayTimeAndDispose();
                fullViewSolution = doubleSolver.solveView(conditions, new ArrayList<>(conditionRegions), consistencyConstraints);
                debugSolvingErrorPerCondition(conditions, fullViewSolution, sortedViewColumns);
                break;

            default:
                throw new RuntimeException("Unsupported SolverType " + solverType);
        }
        return fullViewSolution;
    }
    
    public Set<FormalConditionOr> getProjectedConditions(Set<String> commonAttribsSet, List<List<FormalCondition>> conditionsInSOPForm, List<boolean[]> allTrueBS, Map<String, IntList> bucketFloorsByColumns, List<String> sortedViewColumns) {
    	List<boolean[]> allTrueBSofCommonAttribs = new ArrayList<>();
    	List<boolean[]> allTrueBSofNotCommonAttribs = new ArrayList<>();
		Map<String, IntList> bucketFloorsByColumnsOfCommonAttribs = new HashMap<>();
		Map<String, IntList> bucketFloorsByColumnsOfNotCommonAttribs = new HashMap<>();
		
		List<String> commonAttribs = new ArrayList<>(commonAttribsSet);
		Collections.sort(commonAttribs);
		
		List<String> notCommonAttribs = new ArrayList<>(sortedViewColumns);
		notCommonAttribs.removeAll(commonAttribsSet);
		Collections.sort(notCommonAttribs);
		
		for (int k = 0; k < sortedViewColumns.size(); ++k) {
			String column = sortedViewColumns.get(k);
			if(commonAttribsSet.contains(column)) {
				allTrueBSofCommonAttribs.add(allTrueBS.get(k));
				bucketFloorsByColumnsOfCommonAttribs.put(column, bucketFloorsByColumns.get(column));
			}
			else {
				allTrueBSofNotCommonAttribs.add(allTrueBS.get(k));
				bucketFloorsByColumnsOfNotCommonAttribs.put(column, bucketFloorsByColumns.get(column));
			}
		}
    	
    	Set<FormalConditionOr> newConsistencyConstraints = new HashSet<>();
    	
    	for (List<FormalCondition> condition : conditionsInSOPForm) {	// For each condition
    		HashMap<FormalConditionAnd, FormalConditionOr> conditionMap = new HashMap<>();
    		for (FormalCondition subcondition : condition) {
    			FormalConditionAnd newConditionKept = new FormalConditionAnd();	// part od subcondition on common attribs
    			FormalConditionAnd newConditionDropped = new FormalConditionAnd();	// part of subcondition not on common attribs
    			if (subcondition instanceof FormalConditionAnd) {
    				for (FormalCondition simpleCondition : ((FormalConditionAnd)subcondition).getConditionList()) {
    					if(commonAttribsSet.contains(((FormalConditionSimpleInt)simpleCondition).getColumnname())) {
    						FormalCondition temp = simpleCondition.getDeepCopy();
    						temp.setOutputCardinality(-1);		// Make hashing consistent
    						newConditionKept.addCondition(temp);
    					}
    					else {
    						FormalCondition temp = simpleCondition.getDeepCopy();
    						temp.setOutputCardinality(-1);
    						newConditionDropped.addCondition(temp);
    					}
    		        }
                }
    			else if (subcondition instanceof FormalConditionSimpleInt) {
    				if(commonAttribsSet.contains(((FormalConditionSimpleInt)subcondition).getColumnname())) {
    					FormalCondition temp = subcondition.getDeepCopy();
						temp.setOutputCardinality(-1);		// Make hashing consistent
						newConditionKept.addCondition(temp);
					}
                }
    			else throw new RuntimeException("Unsupported FormalCondition of type" + subcondition.getClass() + " " + subcondition);
    			if(newConditionKept.size() != 0) {
    				if(!conditionMap.containsKey(newConditionKept))
    					conditionMap.put(newConditionKept, new FormalConditionOr());
	    			conditionMap.get(newConditionKept).addCondition(newConditionDropped);
    			}
			}
    		
    		/*
//    		Remove subsets
    		List<FormalConditionAnd> conditionsToRemove = new ArrayList<>();
    		for (FormalConditionAnd outer : conditionMap.keySet()) {
    			for (FormalConditionAnd inner : conditionMap.keySet()) {
    				if(outer == inner || conditionsToRemove.contains(inner))
    					continue;
    				if(isProperSubsetOf(inner, outer, commonAttribs, allTrueBSofCommonAttribs, bucketFloorsByColumnsOfCommonAttribs)) {
    					conditionsToRemove.add(inner);
    				}
    			}
			}
    		for (FormalConditionAnd key : conditionsToRemove) {
				conditionMap.remove(key);
			}*/
    		
    		
    		HashMap<CustomBoolean, List<FormalConditionAnd>> areAllTrueToListKeptCondition = new HashMap<>();
    		for (Entry<FormalConditionAnd, FormalConditionOr> entry : conditionMap.entrySet()) {
    			FormalConditionAnd keptPart = entry.getKey();
				FormalConditionOr droppedPart = entry.getValue();
				
				for (FormalCondition subDroppedPart : droppedPart.getConditionList()) {
					List<boolean[]> bsCopyDroppedPart = deepCopyBS(allTrueBSofNotCommonAttribs);
			        setFalseAppropriateBuckets(subDroppedPart, notCommonAttribs, bsCopyDroppedPart, bucketFloorsByColumnsOfNotCommonAttribs);
			        CustomBoolean areAllTrue = new CustomBoolean(notCommonAttribs.size());
					for (int i = 0; i < bsCopyDroppedPart.size(); i++) {
						areAllTrue.set(i, getAreAllTrue(bsCopyDroppedPart.get(i)));
					}
					if(!areAllTrueToListKeptCondition.containsKey(areAllTrue))
						areAllTrueToListKeptCondition.put(areAllTrue, new ArrayList<>());
					areAllTrueToListKeptCondition.get(areAllTrue).add(keptPart);
				}		        
			}
    		for (Entry<CustomBoolean, List<FormalConditionAnd>> entry : areAllTrueToListKeptCondition.entrySet()) {
				List<FormalConditionAnd> keptPartList = entry.getValue();
				FormalConditionOr conditionToAdd = new FormalConditionOr();
				for (FormalConditionAnd formalConditionAnd : keptPartList) {
					conditionToAdd.addCondition(formalConditionAnd);
				}
				newConsistencyConstraints.add(conditionToAdd);
			}
    	}
        return newConsistencyConstraints;
    }

    private boolean getAreAllTrue(boolean[] bs) {
		for (boolean b : bs) {
			if(!b)
				return false;
		}
		return true;
	}

    /*
//    is A proper subset of B
    private boolean isProperSubsetOf(FormalConditionAnd A, FormalConditionAnd B, List<String> commonAttribs, List<boolean[]> truncatedAllTrueBS, Map<String, IntList> truncatedBucketFloorsByColumns) {
    	boolean properSubset = false;
    	List<boolean[]> bsCopyA = deepCopyBS(truncatedAllTrueBS);
        setFalseAppropriateBuckets(A, commonAttribs, bsCopyA, truncatedBucketFloorsByColumns);
        List<boolean[]> bsCopyB = deepCopyBS(truncatedAllTrueBS);
        setFalseAppropriateBuckets(B, commonAttribs, bsCopyB, truncatedBucketFloorsByColumns);
        for (int i = 0; i < bsCopyA.size(); i++) {
        	boolean[] bucket_i_ofA = bsCopyA.get(i);
        	boolean[] bucket_i_ofB = bsCopyB.get(i);
			for (int j = 0; j < bucket_i_ofA.length; j++) {
				if(bucket_i_ofA[j] && !bucket_i_ofB[j])
					return false;
				if(!bucket_i_ofA[j] && bucket_i_ofB[j])
					properSubset = true;
			}
		}
        if(properSubset)
        	return true;
        else
        	return false;
    }*/

    private static void debugSolvingErrorPerCondition(List<FormalCondition> conditions, ViewSolution viewSolution, List<String> sortedColumns) {
        if (!DebugHelper.solvingErrorCheckNeeded())
            return;

        DebugHelper.printDebug("Evaluating sampling/merging errors per condition");
        ConditionsEvaluator.debugErrorPerConditionWithException(conditions, viewSolution, sortedColumns);
    }

    //    private void exportToFile(String viewname, Set<String> nonKeys, CliqueSolutionInMemory cliqueSolution) {
    //
    //        try {
    //            List<String> sortedColumnnames = new ArrayList<>(nonKeys);
    //            Collections.sort(sortedColumnnames);
    //
    //            List<String> chosenColumns = new ArrayList<>();
    //            for (int i : cliqueSolution.getColIndexes()) {
    //                chosenColumns.add(sortedColumnnames.get(i));
    //            }
    //
    //            FileWriter fw = new FileWriter(new File("/home/dsladmin/CODD/RaghavSood/intermediateViewSolution", viewname));
    //            String header = "COUNT," + StringUtils.join(chosenColumns, ",") + "\n";
    //            fw.write(header);
    //
    //            for (ValueCombination valueCombination : cliqueSolution.getValueCombinations()) {
    //                fw.write(valueCombination.toStringFileDump() + "\n");
    //            }
    //
    //            fw.close();
    //        } catch (IOException ex) {
    //            throw new RuntimeException("File writing error", ex);
    //        }
    //
    //    }

    private List<Region> getConditionRegions(List<List<FormalCondition>> conditionsInSOPForm, List<boolean[]> allTrueBS, List<String> sortedColumns,
            Map<String, IntList> bucketFloorsByColumns) {
        List<Region> conditionRegions = new ArrayList<>();
        for (int i = 0; i < conditionsInSOPForm.size(); i++) {
//            FormalCondition condition = conditions.get(i);
//            List<FormalCondition> subconditions = getSOPSubconditions(condition);
        	List<FormalCondition> subconditions = conditionsInSOPForm.get(i);
            Region conditionRegion = new Region();
            for (FormalCondition subcondition : subconditions) {	// which buckets follow this particular subcondition
                List<boolean[]> bsCopy = deepCopyBS(allTrueBS);

                //Unmarking false buckets
                setFalseAppropriateBuckets(subcondition, sortedColumns, bsCopy, bucketFloorsByColumns);//assert: every element of bucketStructures has atleast one true entry

                BucketStructure subConditionBS = new BucketStructure();
                for (int j = 0; j < bsCopy.size(); j++) {
                    Bucket bucket = new Bucket();
                    for (int k = 0; k < bsCopy.get(j).length; k++) {
                        if (bsCopy.get(j)[k]) {
                            bucket.add(k);	// What split points of this dimension follow this sub constraint
                        }
                    }
                    subConditionBS.add(bucket);	// For particular dimension
                }
                conditionRegion.add(subConditionBS);	// For every subcondition in condition
            }
            conditionRegions.add(conditionRegion);	// For every condition
        }
        return conditionRegions;
    }

    private List<FormalCondition> getSOPSubconditions(FormalCondition condition) {
        FormalConditionSOP sopCondition = new FormalConditionSOP(condition);
        return sopCondition.getConditionList();
    }

    private static void setFalseAppropriateBuckets(FormalCondition condition, List<String> sortedColumns, List<boolean[]> bucketStructures,
            Map<String, IntList> bucketFloorsByColumns) {

        if (condition instanceof FormalConditionAnd) {
            setFalseAppropriateBuckets((FormalConditionAnd) condition, sortedColumns, bucketStructures, bucketFloorsByColumns);
            return;
        }
        if (condition instanceof FormalConditionSimpleInt) {
            setFalseAppropriateBuckets((FormalConditionSimpleInt) condition, sortedColumns, bucketStructures, bucketFloorsByColumns);
            return;
        }
        throw new RuntimeException("Unsupported FormalCondition of type" + condition.getClass() + " " + condition);
    }

    private static void setFalseAppropriateBuckets(FormalConditionAnd andCondition, List<String> sortedColumns, List<boolean[]> bucketStructures,
            Map<String, IntList> bucketFloorsByColumns) {
        for (FormalCondition condition : andCondition.getConditionList()) {
            setFalseAppropriateBuckets(condition, sortedColumns, bucketStructures, bucketFloorsByColumns);
        }
    }

    private static void setFalseAppropriateBuckets(FormalConditionSimpleInt simpleCondition, List<String> sortedColumns, List<boolean[]> bucketStructures,
            Map<String, IntList> bucketFloorsByColumns) {

        String columnname = simpleCondition.getColumnname();
        int columnIndex = sortedColumns.indexOf(columnname);
        boolean[] bucketStructure = bucketStructures.get(columnIndex);
        IntList bucketFloors = bucketFloorsByColumns.get(columnname);

        Symbol symbol = simpleCondition.getSymbol();
        long val = simpleCondition.getValue();

        switch (symbol) {
            case LT:
                for (int i = bucketFloors.size() - 1; i >= 0 && bucketFloors.getInt(i) >= val; i--) {
                    bucketStructure[i] = false;
                }
                break;
            case LE:
                for (int i = bucketFloors.size() - 1; i >= 0 && bucketFloors.getInt(i) > val; i--) {
                    bucketStructure[i] = false;
                }
                break;
            case GT:
                for (int i = 0; i < bucketFloors.size() && bucketFloors.getInt(i) <= val; i++) {
                    bucketStructure[i] = false;
                }
                break;
            case GE:
                for (int i = 0; i < bucketFloors.size() && bucketFloors.getInt(i) < val; i++) {
                    bucketStructure[i] = false;
                }
                break;
            case EQ:
                for (int i = 0; i < bucketFloors.size(); i++) {
                    if (bucketFloors.getInt(i) != val) {
                        bucketStructure[i] = false;
                    }
                }
                break;
            default:
                throw new RuntimeException("Unrecognized Symbol " + symbol);
        }
    }

    private static List<boolean[]> deepCopyBS(List<boolean[]> bucketStructures) {
        List<boolean[]> bucketStructuresCopy = new ArrayList<>();
        for (boolean[] bucketStructure : bucketStructures) {
            boolean[] bucketStructureCopy = Arrays.copyOf(bucketStructure, bucketStructure.length);
            bucketStructuresCopy.add(bucketStructureCopy);
        }
        return bucketStructuresCopy;
    }
}

class CustomBoolean {
	public static final int SEED = 173;
	public static final int PRIME = 37;
	
	boolean booleanArray[];
	
	public void set(int i, boolean data) {
		booleanArray[i] = data;
	}
	
	public boolean get(int i) {
		return booleanArray[i];
	}
	
	public CustomBoolean(int size) {
		booleanArray = new boolean[size];
	}
	
	public int hash(int seed, boolean aBoolean) {
		return (PRIME * seed) + (aBoolean ? 1231 : 1237);
	}

	public int hash(int seed) {
		if (booleanArray == null) {
			return 0;
		}
		for (boolean aBoolean : booleanArray) {
			seed = hash(seed, aBoolean);
	    }
	    return seed;
	}
	  
	@Override
	public boolean equals(Object obj) {
	    if (obj == null) {
	        return false;
	    }
	    if (!CustomBoolean.class.isAssignableFrom(obj.getClass())) {
	        return false;
	    }
	    final CustomBoolean other = (CustomBoolean) obj;
	    if(this.hashCode() == other.hashCode())
	    	return true;
	    else
	    	return false;
	}

	@Override
	public int hashCode() {
		return hash(SEED);
	}
}