package iisc.dsl.coddgen.ui.utils;

import java.awt.Color;
import java.awt.EventQueue;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JTextArea;
import javax.swing.border.BevelBorder;

/**
 * @author adapted from http://stackoverflow.com/a/343007/2202712
 *
 */
public class ConsoleOutputStream {

    private static final String NEWLINE = "\n";
    private final JTextArea     console;

    // *************************************************************************************************
    // INSTANCE MEMBERS
    // *************************************************************************************************

    private Appender            appender;      // most recent action

    public ConsoleOutputStream() {
        this(300);
    }

    public ConsoleOutputStream(int maxlines) {
        if (maxlines < 1)
            throw new IllegalArgumentException("TextAreaOutputStream maximum lines must be positive (value=" + maxlines + ")");

        console = new JTextArea();
        console.setBorder(BorderFactory.createBevelBorder(BevelBorder.LOWERED));
        console.setBackground(Color.WHITE);
        console.setEditable(false);
        console.setLineWrap(true);

        appender = new Appender(console, maxlines);
    }

    public JTextArea getConsole() {
        return console;
    }

    /** Clear the current console text area. */
    public synchronized void clear() {
        if (appender == null)
            throw new RuntimeException("ConsoleOutputStream already closed");
        appender.clear();
    }

    public synchronized void close() {
        appender = null;
    }

    public synchronized void flush() {}

    public synchronized void write(String str) {
        if (appender == null)
            throw new RuntimeException("ConsoleOutputStream already closed");
        if (!str.endsWith(NEWLINE)) {
            str = str + NEWLINE;
        }
        appender.append(str);
    }

    // *************************************************************************************************
    // STATIC MEMBERS
    // *************************************************************************************************

    static class Appender implements Runnable {
        private final JTextArea           console;
        private final int                 maxLines;  // maximum lines allowed in text area
        private final LinkedList<Integer> lengths;   // length of lines within text area
        private final List<String>        values;    // values waiting to be appended

        private int                       curLength; // length of current line
        private boolean                   clear;
        private boolean                   queue;

        Appender(JTextArea console, int maxLines) {
            this.console = console;
            this.maxLines = maxLines;
            lengths = new LinkedList<>();
            values = new ArrayList<>();

            curLength = 0;
            clear = false;
            queue = true;
        }

        synchronized void append(String val) {
            values.add(val);
            //lengths.add(val.length());
            if (queue) {
                queue = false;
                EventQueue.invokeLater(this);
            }
        }

        synchronized void clear() {
            clear = true;
            curLength = 0;
            lengths.clear();
            values.clear();
            if (queue) {
                queue = false;
                EventQueue.invokeLater(this);
            }
        }

        // MUST BE THE ONLY METHOD THAT TOUCHES textArea!
        @Override
        public synchronized void run() {
            if (clear) {
                console.setText("");
            }
            for (String val : values) {
                curLength += val.length();
                if (val.endsWith(EOL1) || val.endsWith(EOL2)) {
                    if (lengths.size() >= maxLines) {
                        console.replaceRange("", 0, lengths.removeFirst());
                    }
                    lengths.addLast(curLength);
                    curLength = 0;
                }
                console.append(val);
            }
            values.clear();
            clear = false;
            queue = true;
        }

        static private final String EOL1 = "\n";
        static private final String EOL2 = System.getProperty("line.separator", EOL1);
    }

} /* END PUBLIC CLASS */
