/***************** svector: A sorted vector of Items *******************
 The class svector is a wrapper around the template vector class of the
 Standard Template Library (STL).  The main modification is that the
 member functions insert() and remove() will do their tasks maintaining
 the vector in increasing order w.r.t the < operator.  Hence that
 operator should have been defined for the contents of the vector. 

 Operators += and -= are provided to perform union and difference of
 two svectors.  A shrink() member function is provided to remove any
 extra space that is being used by the svector.

 The includes() function is provided to determine if one svector
 includes another (i.e. the latter is a subset of the former).

 Finally i/o operators are provided which rely on the fact that the i/o
 operators of the contents of the svector have been defined.

    Copyright (C) 2003 Database Systems Lab, SERC, IISc, Bangalore.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***********************************************************************/
#ifndef SVECTOR_H_
#define SVECTOR_H_

#include "vector.h"
#include "algo.h"

template<class C>
class svector : public vector<C>
{
public:
//    typedef C T;
    typedef vector<C>::iterator iterator;
    typedef vector<C>::const_iterator const_iterator;
    typedef vector<C>::size_type size_type;
    svector() { }
    svector(const_iterator f,const_iterator l) :
	    vector<C>(f,l) { }
//--------------- add/remove itemsets -----------
    svector<C>& operator+=(const svector<C>& i)
    {
	svector<C> result;
	result.reserve(size()+i.size());
	set_union(begin(),end(),i.begin(),i.end(), back_inserter(result));
	swap(result);
	return *this;
    }

    svector<C>& operator-=(const svector<C>& i)
    {
	svector<C> result;
	result.reserve(size());
	set_difference(begin(),end(),i.begin(),i.end(),back_inserter(result));
	swap(result);
	return *this;
    }
//--------------- add/remove items -----------
    iterator insert(iterator pos,const C& x)
    	{ return(vector<C>::insert(pos,x)); }
    void insert(iterator pos,const_iterator f,const_iterator l)
    	{ vector<C>::insert(pos,f,l); }
    void insert(iterator pos,size_type n, const C& x)
    	{ vector<C>::insert(pos,n,x); }
    svector<C>& insert(const C& c)
    {
	iterator pos;
	pos = lower_bound(begin(),end(),c); //binary search for c
	if (pos == end() || *pos != c)
	    vector<C>::insert(pos,c);
	return *this;
    }

    svector<C>& remove(const C& c)
    {
	iterator pos;
	pos = lower_bound(begin(),end(),c); //binary search for c
	if (pos != end() && *pos == c)
	    erase(pos);
	return *this;
    }
//--------------- find items -----------
    bool includes(const C& c) const
    {
	const_iterator pos;
	pos = lower_bound(begin(),end(),c); //binary search for c
	if (pos == end() || *pos != c)
	    return false;
	return true;
    }

//--------------- save memory ------------
    void shrink()
    { /* Uses fact that copy constructor, doesn't allocate more space
	than necessary. I don't know if this is true in general. */
	vector<C> result(*this);
	swap(result);
    }

//--------------- read from IBM synthetic database ------------
    //WARNING: Do not use this Itemset for any other purpose, if using
    //it for reading transactions from a database. Also, destroy this
    //Itemset using the careful_destroy function below, when it goes out
    //of scope.
    void database_read(istream& s)
    {
	int j, noItems;

	//read in binary mode: SPARCompiler Library Reference Manual
	s.read((char*)&j, sizeof(j)); //skip transaction number
	if ( ! s )
	    return;

	s.read((char*)&j, sizeof(j)); //skip customer number
	s.read((char*)&noItems, sizeof(noItems));

	clear();
	reserve(noItems);
	s.read(reinterpret_cast<char*>(_M_start), noItems*sizeof(Item));
	_M_finish = _M_start + noItems;
    }

    void from_int_array(int *ptr, int noItems)
    {
	_M_start = reinterpret_cast<C*>(ptr);
	_M_finish = _M_start + noItems;
    }
};

//**************** Helper functions ************************/

//-------------------- +/- operations -----------------
template <class C>
inline svector<C> operator+(const svector<C>& i1, const svector<C>& i2)
{
    svector<C> sum(i1);
    sum += i2;
    return sum;
}

template <class C>
inline svector<C> operator-(const svector<C>& i1, const svector<C>& i2)
{
    svector<C> diff(i1);
    diff -= i2;
    return diff;
}

//----------------- set comparison operations ---------

template <class C>
inline bool includes(const svector<C>& i1, const svector<C>& i2)
	{ return (includes(i1.begin(),i1.end(),i2.begin(),i2.end())); }
/* NOTE: The SGI implementation of includes() differs from what
  Stroustrup says in section 18.7.5, 3ed. To cope with the
  implementation, we assume includes() to mean "Does i1 include i2?" */

//--------------------- i/o operations ----------------
template <class C>
ostream& operator<<(ostream& s, const svector<C>& i)
{
//    s << i.size() << " :";
    for (int j = 0; j < (int)i.size(); j++)
	s << i[j] << " ";
//	s << " " << i[j];
    return s;
}

template <class C>
istream& operator>>(istream& s, svector<C>& i)
{
    char skip;
    size_t noItems;
    C item;
    s >> noItems;
    i.reserve(noItems); //reserve space in itemset

    s >> skip; //skip ':'
    for (size_t j = 0; j < noItems; j++)
    	{ s >> item; i.insert(item); }
    return s;
}

// template <class C>
// istream& operator>>(istream& s, svector<C>& i)
// {
//     char skip;
//     int noItems;
//     C item;
//     s >> noItems;
//     if (noItems <= 0)
//     {
// 	noItems *= -1;
// 	s >> skip; //skip :
// 	int count;
// 	s >> count;
// 	s >> skip; //skip :
// 	for (int j = 0; j < noItems; j++)
// 	    s >> count;
// 	i.resize(0);
// 	return s;
//     }
// 
//     i.reserve(noItems); //reserve space in itemset
// 
//     s >> skip; //skip ':'
//     for (size_t j = 0; j < noItems; j++)
//     	{ s >> item; i.insert(item); }
//     return s;
// }

#endif
