#ifndef _RAM_STORAGE_H_
#define _RAM_STORAGE_H_

/***********************************************************************
 AUTHOR: Srikanta Bedathur
 DESCRIPTION:
    RAM (in-memory) storage class - Extension to the abstract Storage.

    Copyright (C) 2004 Database Systems Lab, Supercomputer Education and
    Research Centre, Indian Institute of Science, Bangalore, INDIA.
                     http://dsl.serc.iisc.ernet.in

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***********************************************************************/
#include <string>
#include <ext/hash_map>

#include "options.h"
#include "defines.h"
#include "page.h"
#include "malloc.h"

using namespace std;
#if(__GNUC__==3)
using namespace __gnu_cxx;
#endif

struct eq_SFXULONG {
  bool operator () (const SFXULONG& one,
		    const SFXULONG& two) const{
    return (one == two);
  }
};

typedef hash_map <SFXULONG, page_t*, hash<SFXULONG>, eq_SFXULONG> disk_hash;

class RamStorage : public Storage{
private:
  int fd[4];
  disk_hash *files;

public:
  ///////////////////////
  // Constructor
  ///////////////////////
  RamStorage (bool _isleaf, enum PHASE _phase = CONSTRUCT) : Storage (_isleaf,_phase) {
    files = new disk_hash (104020);
  }

  ///////////////////////
  // Destructor
  ///////////////////////
  ~RamStorage () {
    for (disk_hash::iterator iter = files -> begin();
	 iter != files -> end();
	 iter++) {
      free ((*iter).second);
    }
    files -> clear();
  }
  
  ////////////////////////
  // Open 
  ////////////////////////
  int s_open (SFXULONG fileid) {
    return 1;
  }
  
  ////////////////////////
  // Write the given page
  ////////////////////////
  int s_pwrite (SFXULONG fileid,
		SFXULONG pageid,
		page_t* page_ptr) {
    ASSERT (phase != SEARCH);
    
    disk_hash::iterator iter = files -> find (pageid);
#ifdef FILESTATS
    if (isleaf) {
      write_leaf_count++;
    }
    else {
      write_internal_count++;
    }
#endif
    if (iter == files -> end() ) {
      page_t* page = new page_t(*page_ptr);
      pair <disk_hash::iterator, bool> retval = 
	files -> insert (disk_hash::value_type (pageid, page));
#ifdef DEBUG
      if (retval.second == false) {
	// it was not inserted...
	cerr<< "ERROR ..... COULD NOT WRITE THE PAGE" << endl;
	cerr << "fileid = " << fileid 
	     << " pageid = " << pageid 
	     << " isleaf = " << isleaf
	     << endl;
      }
      else {
	DBG_TRACE("Inserting : (" 
		  << fileid << ", " 
		  << pageid << ", " 
		  << isleaf << ")" << endl);
      }
      ASSERT (retval.second == true);
#endif
    }
    else {
      DBG_TRACE("Found : (" 
		<< fileid << ", " 
		<< pageid << ", " 
		<< isleaf << endl);
      page_t* old = ((*iter).second);
      *old = *page_ptr;
    }
    return DISKPAGESIZE;
  }
  
  /////////////////////////
  // Read in the page.
  /////////////////////////
  int s_pread (SFXULONG fileid,
	       SFXULONG pageid,
	       page_t* page_ptr) {
#ifdef FILESTATS
    if (isleaf) {
      read_leaf_count++;
    }
    else{
      read_internal_count++;
    }
#endif
    disk_hash::iterator temp = files -> find (pageid);
    ASSERT (temp != files -> end());
    *page_ptr = *((*temp).second);
    return DISKPAGESIZE;
  }

};



#endif
