#ifndef _S_MAN_H_
#define _S_MAN_H_

/***********************************************************************
 AUTHOR: Srikanta Bedathur
 DESCRIPTION:
    Storage Manager base class.

    Copyright (C) 2004 Database Systems Lab, Supercomputer Education and
    Research Centre, Indian Institute of Science, Bangalore, INDIA.
                     http://dsl.serc.iisc.ernet.in

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***********************************************************************/
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include <queue>
#include <vector>
#include <ext/hash_map>

#include "malloc.h"
#include "storage.h"
#include "defines.h"
#include "page.h"
#include "key.h"
#include "s_pagerec.h"

using namespace std;
#if(__GNUC__==3)
using namespace __gnu_cxx;
#endif



/* Hashtable of pages */
struct eq_pagehash{
  bool operator () (const SFXULONG& one,
		    const SFXULONG& two) const {
    return (one == two);
  }
};

typedef hash_map <SFXULONG, pagerec_t*, hash <SFXULONG>, eq_pagehash> page_hash;



////////////////////////////////////////////////////////////////
/// Storage Manager is in charge of buffer management policy, 
/// and uses Storage class and provides services to the 
/// Allocator
////////////////////////////////////////////////////////////////
template <class A, bool isleaf>
class StorageManager {

protected:

  page_hash buffer_table;
  
  long MAXBUFFER;

  Storage *_storage;

public:

  //////////////////////////////////////////////////
  //  Constructor
  /////////////////////////////////////////////////

  StorageManager (Storage *storage,
		  long buffpoolsize, 
		  enum PHASE _phase = CONSTRUCT) :
    buffer_table (buffpoolsize + 1) {
    _storage = storage;
    MAXBUFFER = buffpoolsize;
  }

  /////////////////////////////////////////////////
  /// Destructor
  /////////////////////////////////////////////////

  ~StorageManager () {
    DBG_SM_TRACE("StorageManager::~StorageManager()" << endl);
    for (page_hash::iterator temp = buffer_table.begin();
	 temp != buffer_table.end();
	 temp++) {
      
      writeback ((*temp).second);
      
      delete ((*temp).second->page_ptr);
      delete ((*temp).second);
    }
    buffer_table.clear();
  }

  ////////////////////////////////////////////////
  //// Pinning and unpinning operations
  ////////////////////////////////////////////////
  
  void pin (SFXULONG fileid,
	    SFXULONG pageid,
	    page_t* page) {
    DBG_SM_TRACE ("SMan::pin( " << fileid << ", " << pageid << ")" << endl);
    ASSERT (page);
    SFXULONG key;
    KEY(key, fileid, pageid);
    
    bool insertflag = 
      (buffer_table.insert((page_hash::value_type 
			    (key, new pagerec_t (page,key,true,true))))).second;

    assert (insertflag);

    _storage->s_open (fileid);
  }


  void unpin (SFXULONG fileid,
	      SFXULONG pageid) {
    DBG_SM_TRACE ("SMan::unpin( " << fileid << ", " << pageid << ")" << endl);
    
    SFXULONG key;
    KEY(key, fileid, pageid);
    page_hash::iterator temp = buffer_table.find (key);
    
#ifdef FILESTATS
    if (!isleaf) {
      unpin_internal_count++;
    }
    else {
      unpin_leaf_count++;
    }
#endif
    
    (*temp).second -> beingwritten = false;
    (*temp).second -> dirtyflag = true;

    makespace ();

    insert ((*temp).second);
    
  }

  /////////////////////////////////////////////////////////
  /// Fetch the requested node - from disk if needed - and
  /// return the pointer
  ////////////////////////////////////////////////////////

  A* get_obj (SFXULONG fileid,
	      SFXULONG pageid,
	      SFXULONG recid) {
    DBG_SM_TRACE ("get_obj " << isleaf << " - " << fileid << ", " << pageid << ", " << recid << endl);

    SFXULONG key;
    KEY(key, fileid, pageid);
    page_hash::iterator temp = buffer_table.find (key);
    
    if (temp == buffer_table.end()) {
      /* The requested page is not in the buffer */
#ifdef SMSTATS
      if (!isleaf) {
	buffer_internal_requests++;
      }
      else {
	buffer_leaf_requests++;
      }
#endif     
     
      makespace ();

      page_t* page = new page_t();

      _storage->s_pread (fileid,
			 pageid,
			 page);
      
      temp = (buffer_table.insert 
	      (page_hash::value_type 
	       (key, new pagerec_t (page,key)))).first;
      
      insert ((*temp).second);
    }
    else {
      /* required page in the buffer */
      if (!(*temp).second -> beingwritten) {
#ifdef SMSTATS
	if (!isleaf) {
	  buffer_internal_requests++;
	  buffer_internal_hits++;
	}
	else {
	  buffer_leaf_requests++;
	  buffer_leaf_hits++;
	}
#endif 
	reposition((*temp).second); 
      }
    }
    
    (*temp).second -> ref_count ++;
    return ((A*)((*temp).second -> page_ptr -> data() + 
		 (recid * sizeof (A))));
  }

  //////////////////////////////////////////////////////////
  /// Reference management onto the pages
  //////////////////////////////////////////////////////////
  void inc_ref (SFXULONG fileid,
		SFXULONG pageid) {
    DBG_SM_TRACE ("SMan::inc_ref( " << fileid << ", " << pageid << ")" << endl);
    
    SFXULONG key;
    KEY(key, fileid, pageid);
    page_hash::iterator temp = buffer_table.find (key);
    (*temp).second -> ref_count += 1;
  }

  void dec_ref (SFXULONG fileid,
		SFXULONG pageid) {
    DBG_SM_TRACE ("SMan::dec_ref( " << fileid << ", " << pageid << ")" << endl);

    SFXULONG key;
    KEY(key, fileid, pageid);
    page_hash::iterator temp = buffer_table.find (key);
    (*temp).second -> ref_count -= 1;
  }

  //////////////////////////////////////////////////////////
  /// Set the dirty status of the page
  //////////////////////////////////////////////////////////
  void dirtify (SFXULONG fileid, 
		SFXULONG pageid) {
    DBG_SM_TRACE ("SMan::dirtify( " << fileid << ", " << pageid << ")" << endl);

    SFXULONG key;
    KEY(key,fileid,pageid);
    
    (*(buffer_table.find (key))).second->dirtyflag = true;
  }



  /////////////////////////////////////////////////////////
  ////    Cleanup the buffer pool
  //// This method useful when construction is immediately
  //// followed by searching.
  /////////////////////////////////////////////////////////
  virtual void cleanup () {
    int i =0;
    vector <page_hash::key_type> erase_list;

    for (page_hash::iterator temp = buffer_table.begin();
	 temp != buffer_table.end();
	 temp++,i++) {
      writeback ((*temp).second);
      if ((*temp).second -> ref_count <= 0) {
	assert ((*temp).second->page_ptr);
	delete ((*temp).second->page_ptr);
	delete ((*temp).second);
	erase_list.push_back((*temp).first);
      }
    }
    for (vector<page_hash::key_type>::iterator eraser = erase_list.begin();
	 eraser != erase_list.end();
	 eraser++) {
      buffer_table.erase (*eraser);
    }
  }


protected:
  //////////////////////////////////////////////////////////
  //// Ask storage to write the page into the disk image and
  //// reset its dirty flag.
  //////////////////////////////////////////////////////////
  void writeback (pagerec_t* page) {
    DBG_SM_TRACE ("SMan::writeback( " << FILEID(page -> key) << ", " << PAGEID(page -> key) << ")" << endl);

    ASSERT (page);
    ASSERT (page->page_ptr);
    if (page -> dirtyflag) {
      _storage->s_pwrite (FILEID(page -> key),
			  PAGEID(page -> key),
			  page -> page_ptr);
      page -> dirtyflag = false;
    }
    return;
  }
  
  /////////////////////////////////////////////////////////
  //// Purge the page from the bufferpool, writing it back
  //// if necessary.
  /////////////////////////////////////////////////////////
  void purge (pagerec_t* page) {
    DBG_SM_TRACE("SMan::purge( " 
	 << FILEID(page->key) 
	 << ", " << PAGEID(page->key) 
	 << ")" << endl);

    ASSERT (page);
    ASSERT (!page -> beingwritten);
    ASSERT (page -> page_ptr);

    DBG_SM_TRACE ("S_Man::purge " << FILEID(page->key) << ", " << PAGEID (page->key) << endl);

    page_hash::iterator temp = buffer_table.find (page -> key);
    ASSERT (temp != buffer_table.end());

    ASSERT ((*temp).second -> page_ptr);
    writeback ((*temp).second);
    ASSERT ((*temp).second -> page_ptr);
    delete ((*temp).second -> page_ptr);
    delete ((*temp).second);
    buffer_table.erase (temp);
  }

protected:

  /*********************************************************
   * All this policy specific code go out into another
   * place - so that we need to only use that specialization
   * of StorageManager template, according to the 
   * required policy. No need to update the codebase for the
   * whole thing.
   *********************************************************/
  virtual void makespace () = 0;
  virtual void reposition (pagerec_t* freshpage) = 0;
  virtual void insert (pagerec_t* inpage) = 0;
};
#endif /*_S_MAN_H_*/


