#ifndef _S_MAN_LRU_H_
#define _S_MAN_LRU_H_

/***********************************************************************
 AUTHOR: Srikanta Bedathur
 DESCRIPTION:
    Storage Manager extension with LRU buffer management.

    Copyright (C) 2004 Database Systems Lab, Supercomputer Education and
    Research Centre, Indian Institute of Science, Bangalore, INDIA.
                     http://dsl.serc.iisc.ernet.in

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***********************************************************************/
#include "s_man.h"
#include "storage.h"

using namespace std;
#if(__GNUC__==3)
using namespace __gnu_cxx;
#endif

template <class A, bool isleaf>
class LruStorageManager : public StorageManager <A, isleaf> {

protected:
  pagerec_t* lru_head;
  pagerec_t* lru_tail;
  SFXULONG lru_count;

public:
  LruStorageManager (Storage *storage,
		     long buffpoolsize, 
		     enum PHASE _phase = CONSTRUCT) : 
    StorageManager <A, isleaf> (storage, buffpoolsize, _phase) {
    
    lru_head = NULL;
    lru_tail = NULL;
    lru_count = 0;
  }

protected:
  
  void makespace () {
    if (buffer_table.size() < MAXBUFFER)
      return;
    pagerec_t* lru_iter = lru_tail;

    for (; lru_iter != NULL; lru_iter = lru_iter->prev) {
      if (lru_iter->ref_count <= 0) {
	LRU_delete (lru_iter);
	purge (lru_iter);
	return;
      }
    }
    err_msg ("No victim page found for eviction!!", DISASTER);
    return;   
  }

  void insert (pagerec_t* page) {
    if (lru_head == NULL) {
      lru_head = lru_tail = page;
      return;
    }
    ASSERT (lru_tail);
    page -> next = lru_head;
    lru_head -> prev = page;
    lru_head = page;
    lru_count += 1;
  }

  void reposition (pagerec_t* freshpage) {
    ASSERT (lru_head);
    ASSERT (lru_tail);
    ASSERT (freshpage);
    
    if (freshpage == lru_head)
      return;
    if (freshpage == lru_tail) {
      lru_tail = freshpage -> prev;
      ASSERT (lru_tail);
      freshpage -> next = lru_head;
      lru_head -> prev = freshpage;
      lru_head = freshpage;
      freshpage -> prev = NULL;
      return;
    }
    
    freshpage -> prev -> next = freshpage -> next;
    freshpage -> next -> prev = freshpage -> prev;
    freshpage -> next = lru_head;
    lru_head -> prev = freshpage;
    lru_head = freshpage;
    freshpage -> prev = NULL;
  }


  void LRU_delete (pagerec_t* rem) {
    ASSERT (!rem -> beingwritten);
    ASSERT (rem -> ref_count <= 0);

    if (rem == lru_head && rem == lru_tail) {
      lru_head = lru_tail = NULL;
      return;
    }
    
    if (rem == lru_head) {
      lru_head = rem -> next;
      ASSERT (lru_head != NULL);
      lru_head -> prev = NULL;
      rem -> next = NULL;
      rem -> prev = NULL;
      return;
    }
    if (rem == lru_tail) {
      lru_tail = rem -> prev;
      ASSERT (lru_tail != NULL);
      lru_tail -> next = NULL;
      rem -> prev = NULL;
      return;
    }
    
    rem -> prev -> next = rem -> next;
    rem -> next -> prev = rem -> prev;
    rem -> next = rem -> prev = NULL;
    lru_count -= 1;
  }
};





#endif
