/***********************************************************************
 AUTHOR: Srikanta Bedathur
 DESCRIPTION:
    Iterator module for traversing a FASTA -like DNA sequence files.

    Copyright (C) 2004 Database Systems Lab, Supercomputer Education and
    Research Centre, Indian Institute of Science, Bangalore, INDIA.
                     http://dsl.serc.iisc.ernet.in

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***********************************************************************/
#include <assert.h>
#include <string.h>

#include <string>
#include <utility>
#include <iostream>

#include "seqiter.h"



SeqFileIterator::SeqFileIterator( const char *seqFileName ):_seqInputStream(){
  attach( seqFileName );
}



SeqFileIterator::SeqFileIterator():_seqInputStream(){
  _seqFileName[0] = '\0';
  _seqBuffer.resize(0);
}



void SeqFileIterator::attach( const char *seqFileName ){
  
  //check if stream is already open, then close
  if( _seqInputStream.is_open() )
    _seqInputStream.close(); 

  strcpy(_seqFileName,seqFileName);
  _seqInputStream.open(_seqFileName);

  _seqBuffer.resize(0);

}

void SeqFileIterator::init(){
  if( ! _seqInputStream.is_open() ){
    ERRLOG("Unattached Iterator");
    _status = seqFAIL;
    return;
  }
  _status = seqREADY;
}

bool 
SeqFileIterator::eof(){
  if(seqEOF == _status )
    return true;
  else return false;
}

bool
SeqFileIterator::fail() {
  if (seqFAIL == _status)
    return true;
  else return false;
}

int 
SeqFileIterator::size(){
  return _seqBuffer.length();
}

SeqFileIterator::~SeqFileIterator(){
  cerr << "deleting the iterator for file" << _seqFileName << endl;
  if ( _seqInputStream ){
    if( _seqInputStream.is_open() )
      _seqInputStream.close();   
  }
  _seqBuffer.resize(0);
  _metaline.resize(0);
}

void 
SeqFileIterator::close(){
  if (_seqInputStream.is_open()){
    _seqInputStream.close();
  }
  _status = seqCLOSE;
  _seqBuffer.resize(0);
  _metaline.resize(0);
}

#ifdef CLEANUP
string
SeqFileIterator::cleanup(char* cur_str) {
  
  string retval = "";
  
  for (int i = 0; i < strlen(cur_str); i++) {
    if (cur_str[i] == 'a' || cur_str[i] == 'A' ||
	cur_str[i] == 'c' || cur_str[i] == 'C' ||
	cur_str[i] == 'g' || cur_str[i] == 'G' ||
	cur_str[i] == 't' || cur_str[i] == 'T') {
      retval = retval + cur_str[i];
    }
  }
  return retval;
}

#endif


/* Returns a pair of strings - metaline and the sequence itself */
pair<string, string> SeqFileIterator::next() {

  pair <string, string> retval;
  _seqBuffer.resize(0);
  _metaline.resize(0);
  retval.first = _metaline;
  retval.second = _seqBuffer;

  if (seqREADY != _status && seqITER != _status) {
    _seqBuffer.resize(0);
    _metaline.resize(0);
    return retval;
  }
  
  _status = seqITER;

  if (_seqInputStream.eof()) {
    _status = seqEOF;
    _metaline.resize(0);
    _seqBuffer.resize(0);
    return retval;
  }

  char cur_str [STRLENGTH];
  _seqBuffer.resize(0);

  while (!_seqInputStream.eof()) {
    if (_seqInputStream.peek() != '>') {
      if (_metaline.length () == 0) {
	// If there is no metaline associated with the
	// current line
	cerr << "ERROR : where is the metaline in the query file?" << endl;	
	_status = seqFAIL;
	_seqBuffer.resize (0);
      }
      _seqInputStream.getline (cur_str, STRLENGTH);
#ifdef CLEANUP
      _seqBuffer = _seqBuffer + cleanup(cur_str);
#else
      _seqBuffer = _seqBuffer + cur_str;
#endif
    }

    else {
      if (_metaline.length () == 0) {
	_seqInputStream.getline (cur_str, STRLENGTH);
	_metaline = cur_str;
      }
      else {
	break;
      }
    }
  }

  retval.first = _metaline;
  retval.second = _seqBuffer;
  
  return retval;
}
