/***********************************************************************
 AUTHOR: Srikanta Bedathur
 DESCRIPTION:
    The driver program.

    Copyright (C) 2004 Database Systems Lab, Supercomputer Education and
    Research Centre, Indian Institute of Science, Bangalore, INDIA.
                     http://dsl.serc.iisc.ernet.in

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***********************************************************************/
#include <unistd.h>


#include "sfxnode.h"

#include "storage.h"
#include "ram_storage.h"
#include "disk_storage.h"

#include "s_man.h"
#include "s_man_lru.h"
#include "s_man_topq.h"

#include "allocator.h"

#include "construct.h"
#include "search.h"
#include "options.h"
#include "seqiter.h"

Storage *leaf_store, *internal_store;

main (int argc, char* argv[]) {
  string default_settings = "./suffixrc";
  enum PHASE default_phase = BOTH;
  enum POLICY default_policy = TOPQ;
  enum STORAGE default_storage = DISK;
  long default_internal_buffpool_size = 1000;
  long default_leaf_buffpool_size = 1000;
  long default_qlength = 10;
  string default_queryfilename = "query";
  
  string settings = "";
  enum PHASE phase = NONE;
  enum POLICY policy = (enum POLICY) -1;
  enum STORAGE storage = (enum STORAGE) -1;
  long internal_buffpool_size = -1;
  long leaf_buffpool_size = -1;
  long qlength = -1;
  string queryfilename = "";

  int fd;
  int opt;
  long size;

  while (1) {
    opt = getopt (argc, argv, "hf:csbi:l:p:t:q:Q:");
    if (opt == -1) {
      break;
    }
    switch (opt) {
    case 'h':
      cerr << "Usage: " << argv[0] << " [options]" << endl
	   << "Where options are: " << endl
	   << "\t -f <settings> (default=./suffixrc)" << endl
	   << "\t -[csb] (default=b)" << endl
	   << "\t -i <internal_pool_size> (default=1000)" << endl
	   << "\t -l <leaf_pool_size> (default=1000)" << endl
	   << "\t -p <policy=[LRU,TOPQ]> (default=TOPQ)" << endl
	   << "\t -t <storage=[RAM,DISK]> (default=DISK)" << endl
	   << "\t -q <q length> {if policy = TOPQ only} (default=10)" << endl
	   << "\t -Q <queryfilename> (default=query)" << endl
	   << endl;
      return -1;
    case 'f':
      if (optarg[0] == '-') {
	cerr << "ERROR [Param -f]: Invalid Settings filename" << endl;
	exit (-1);
      }
      settings = optarg;
      break;
    case 'c':
      if (phase != NONE) {
	cerr << "ERROR [Param -c]: Only one of -c -s -b allowed" << endl;
	exit (-2);
      }
      phase = CONSTRUCT;
      break;
    case 's':
      if (phase != NONE) {
	cerr << "ERROR [Param -c]: Only one of -c -s -b allowed" << endl;
	exit (-2);
      }
      phase = SEARCH;
      break;
    case 'b':
      if (phase != NONE) {
	cerr << "ERROR [Param -c]: Only one of -c -s -b allowed" << endl;
	exit (-2);
      }
      phase = BOTH;
      break;
    case 'i':
      if (optarg [0] == '-') {
	cerr << "ERROR [Param -i]: Invalid Internal Buffer Pool Size" << endl;
	exit (-3);
      }
      internal_buffpool_size = strtol (optarg, (char**)NULL, 10);
      if (internal_buffpool_size <= 6) {
	cerr << "ERROR: We need at least 6 buffers for Internal nodes" << endl;
	exit (-3);
      }
      break;
    case 'l':
      if (optarg [0] == '-') {
	cerr << "ERROR [Param -i]: Invalid Leaf Buffer Pool Size" << endl;
	exit (-3);
      }

      leaf_buffpool_size = strtol (optarg, (char**)NULL, 10);
      if (leaf_buffpool_size <= 3) {
	cerr << "ERROR: We need at least 3 buffers for Leaf nodes" << endl;
	exit (-4);
      }
      break;
    case 'p':
      if (strcmp(optarg, "LRU") == 0) {
	policy = LRU;
      }
      else if (strcmp(optarg, "TOPQ") == 0) {
	policy = TOPQ;
      }
      else {
	cerr << "ERROR [Param -p]: Invalid Policy name - should be LRU or TOPQ" << endl;
	exit (-5);
      }
      break;
    case 't':
      if (strcmp (optarg, "RAM") == 0) {
	storage=RAM;
      }
      else if (strcmp (optarg, "DISK") == 0) {
	storage = DISK;
      }
      else {
	cerr << "ERROR [Param -t]: Invalid Storage option - should be RAM or DISK" << endl;
	exit (-6);
      }
      break;
    case 'q':
      if (policy != TOPQ) {
	cerr << "ERROR [Param -q]: Specify Q Length after Policy specification" << endl;
	exit (-5);
      }

      if (optarg [0] == '-') {
	cerr << "ERROR [Param -q]: Invalid Q Length" << endl;
	exit (-3);
      }

      qlength = strtol (optarg, (char**)NULL, 10);
      break;
    case 'Q':
      if (phase == NONE || phase == CONSTRUCT) {
	cerr << "ERROR [Param -Q]: Specify queryfilename after specification of search" << endl;
	exit (-7);
      }
      queryfilename = optarg;
      break;
    }
  }    

  if (settings == "") 
    settings = default_settings;

  if (phase == NONE)
    phase = default_phase;

  if (policy == (enum POLICY) -1)
    policy = default_policy;
  
  if (storage == (enum STORAGE) -1)
    storage = default_storage;

  if (internal_buffpool_size == -1)
    internal_buffpool_size = default_internal_buffpool_size;
  
  if (leaf_buffpool_size == -1)
    leaf_buffpool_size = default_leaf_buffpool_size;

  if (qlength == -1)
    qlength = default_qlength;

  if (queryfilename == "") 
    queryfilename = default_queryfilename;


  Options::read (settings);
  Options::sequence = (char*)malloc(sizeof(char) * Options::datasize + 1);
  fd = open (Options::datafile.c_str(), O_RDONLY);
  if ((size = read (fd, Options::sequence, Options::datasize)) < Options::datasize) {
    cerr << "ERROR: The file is shorter than you said!" << endl;
    exit(-1);
  }
  Options::sequence[Options::datasize] = '$'; // the end character.

  if (Options::internalNodeSize != sizeof(SfxInternalNode) ||
      Options::leafNodeSize != sizeof(SfxNode)) {
    cerr << "ERROR : The size of nodes do not match" << endl;
    cerr << "      : Expected (leaf = " << sizeof(SfxNode) 
	 << ", internal=" << sizeof(SfxInternalNode)
	 << ")" << endl;
    cerr << "      : Given (leaf = " << Options::leafNodeSize
	 << ", internal=" << Options::internalNodeSize
	 << ")" << endl;
    exit (-1);
  }


  if (storage == RAM) {
    leaf_store = new RamStorage (true, phase);
    internal_store = new RamStorage (false, phase);
  }
  else if (storage == DISK) {
    leaf_store = new DiskStorage (true, phase);
    internal_store = new DiskStorage (false, phase);
  }

  if (policy == TOPQ) {
    leaf_sm = new TopqStorageManager <SfxNode, true> (leaf_store,
						      leaf_buffpool_size,
						      qlength,
						      phase);
    internal_sm = new TopqStorageManager <SfxInternalNode, false> (internal_store,
								   internal_buffpool_size,
								   qlength,
								   phase);
  }
  else {
    leaf_sm = new LruStorageManager <SfxNode, true> (leaf_store,
						     leaf_buffpool_size,
						     phase);
    internal_sm = new LruStorageManager <SfxInternalNode, false> (internal_store,
								  internal_buffpool_size,
								  phase);
  }

  root = new Ref (0);

  /*&&&&&&&&&&&&&&&&&& CONSTRUCT PHASE &&&&&&&&&&*/
  if (phase == BOTH || phase == CONSTRUCT) {
    leaf_alloc = new Allocator <SfxNode, true> (leaf_sm);
    internal_alloc = new Allocator <SfxInternalNode, false> (internal_sm);
    


    initstats();
    
    constructSuffixTree (leaf_alloc,
			 internal_alloc,
			 *root,
			 Options::sequence,
			 Options::datasize);
    printstats();
    
    delete (leaf_alloc);
    delete (internal_alloc);
    if (phase == BOTH) {
      leaf_sm -> cleanup ();
      internal_sm -> cleanup();
    }
  }

  /*&&&&&&&&&&&&&&&&&& SEARCH PHASE &&&&&&&&&&&&&*/
  if (phase == BOTH || phase == SEARCH) {
    initstats();
    SeqFileIterator query(queryfilename.c_str());
    query.init();
    int count = 0;
    while (!query.eof() && !query.fail()) {
      if (count++ % 100 == 0) {
	printstats (count);
      }
      pair<string,string> qstr = query.next();
      if (qstr.first.length() != 0 &&
	  qstr.second.length() != 0) {
	int retval =searchSuffixTree (leaf_sm, 
				      internal_sm, 
				      *root, 
				      Options::sequence, 
				      qstr.second.c_str(), 
				      Options::datasize,
				      qstr.second.length());
	if (retval > 0) {//if it occurs
	  cerr << "# " << qstr.first << endl;
	}
      }
      count++;
    }
    printstats();
  }
  delete (leaf_sm);
  delete (internal_sm);

  delete (leaf_store);
  delete (internal_store);

}
