/**********************************************************************
    Copyright (C) 2004 Database Systems Lab, Supercomputer Education and
    Research Centre, Indian Institute of Science, Bangalore, INDIA.
    http://dsl.serc.iisc.ernet.in

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***********************************************************************/


/***********************************************************************
 The AprioriGen Function and other functions useful for Apriori like
 algorithms.
 
 The difference between this file and apriori.h is that the functions
 here are meant to work on hashtrees which store both the count and an
 extra partition field for each itemset. (Hence the 'p' in the
 filename.)  This partition field is useful for some algorithms such as
 PARTITION, CARMA etc.

 As expected, to define a functor, we need to define all the functions
 and functors that it uses, before the actual code of this functor. 
 However we think in a top-down approach here.  Therefore to understand
 the code of each functor, read each section from bottom up.
***********************************************************************/
#ifndef APRIORIP_H
#define APRIORIP_H

#include "pItemset.h"

/************************ AprioriGen *****************************/

//Function object to perform an operation during traversal
struct Apriori_gen2 : public unary_function<const pItemset&, void>
{
    const pHashtree *L;
    pHashtree *C;
    const Itemset *i1;
    int partition;

    Apriori_gen2(const pHashtree *l, pHashtree *c,const Itemset *i,int p)
	    { L=l; C=c; i1=i; partition=p; }

    void operator() (const pItemset& e)
    {
	/* preconditions:
	   1) e.first which is an itemset, must have size>=1
	   2) e.first.size() == i1->size() is true */

	const Itemset& i2 = e.first;
	if (equal(i1->begin(),(i1->end())-1,i2.begin()) &&
		i1->back() < i2.back())
	{
	    Itemset candidate;
	    candidate.reserve(i1->size()+1);
	    candidate.insert(candidate.begin(),i1->begin(),i1->end());
	    candidate.push_back(i2.back());

	    //prune
	    if ( i1->size() > 1 ) {
		Itemset::iterator j;
		for ( j=candidate.begin(); j<candidate.end()-2; j++ ) {
		    Item temp = *j;
		    candidate.erase(j);
		    if ( ! L->contains(candidate) )
			return;
		    candidate.insert(j,temp);
		}
	    }

	    pItemset newEntry;
	    newEntry.first.swap(candidate);
	    newEntry.second.count = 0;
	    newEntry.second.partition = partition;
	    C->move(newEntry);
	}
    }
};

//Function object to perform an operation during traversal
struct Apriori_gen1 : public unary_function<const pItemset&, void>
{
    const pHashtree *L;
    pHashtree *C;
    int partition;

    Apriori_gen1(const pHashtree *l, pHashtree *c,int p)
	    { L=l; C=c; partition=p; }

    void operator() (const pItemset& e)
    {
	//precondition: e.first must have size >= 1

	Itemset i(e.first.begin(), e.first.end()-1);
	for_each_extension(*L, Apriori_gen2(L,C,&(e.first),partition), i,i.size()+1);
    }
};

void AprioriGen(const pHashtree& L, pHashtree& C,int partition)
{
    for_each(L,Apriori_gen1(&L,&C,partition));
}

/************************ NBGen *****************************/

//Function object to perform an operation during traversal
struct NBgen2 : public unary_function<const pItemset&, void>
{
    const pHashtree *L;
    pHashtree *C;
    const Itemset *i1;
    int partition;

    NBgen2(const pHashtree *l, pHashtree *c,const Itemset *i,int p)
	    { L=l; C=c; i1=i; partition=p; }

    void operator() (const pItemset& e)
    {
	/* preconditions:
	   1) e.first which is an itemset, must have size>=1
	   2) e.first.size() == i1->size() is true */

	const Itemset& i2 = e.first;
	if (equal(i1->begin(),(i1->end())-1,i2.begin()) &&
		i1->back() < i2.back())
	{
	    Itemset candidate;
	    candidate.reserve(i1->size()+1);
	    candidate.insert(candidate.begin(),i1->begin(),i1->end());
	    candidate.push_back(i2.back());

	    //prune
	    if (L->contains(candidate))
		return;

	    int maxMissed = -1;
	    if ( i1->size() > 1 ) {
		Itemset::iterator j;
		for ( j=candidate.begin(); j<candidate.end()-2; j++ ) {
		    Item temp = *j;
		    candidate.erase(j);

		    pHashtree::const_iterator sub;
		    sub = L->find(candidate);
		    if (sub == L->end())
			return;

		    int tempmin = sub->second.partition +
			    sub->second.count - 1;
		    if (maxMissed == -1 || tempmin < maxMissed)
			maxMissed = tempmin;

		    candidate.insert(j,temp);
		}
	    }

	    int maxMissed2 = partition;
	    pItemset newEntry;
	    newEntry.first.swap(candidate);
	    newEntry.second.count = 1;
	    newEntry.second.partition = (maxMissed < maxMissed2)?
		    maxMissed : maxMissed2;
	    C->move(newEntry);
	}
    }
};

//Function object to perform an operation during traversal
struct NBgen1 : public unary_function<const pItemset&, void>
{
    const pHashtree *L;
    pHashtree *C;
    int partition;

    NBgen1(const pHashtree *l, pHashtree *c, int p) : L(l),
	    C(c), partition(p) { }

    void operator() (const pItemset& e)
    {
	//precondition: e.first must have size >= 1
	    
	Itemset i(e.first.begin(), e.first.end()-1);
	for_each_extension(*L, NBgen2(L,C,&(e.first),partition), i,i.size()+1);
    }
};

void NBGen(const pHashtree& L, pHashtree& C, int partition)
{
    for_each(L,NBgen1(&L,&C,partition));
}

/************************ IncrCount **************************/
//Function object to perform an operation during traversal
struct IncrCount : public unary_function<pItemset&, void>
{
    void operator() (pItemset& e) { e.second.count++; }
};

/************************ IncrCountSave **********************/
//Function object to perform an operation during traversal
struct IncrCountSave : public unary_function<pItemset&, void>
{
    pHashtree *h;
    IncrCountSave(pHashtree& H) : h(&H) { }
    void operator() (pItemset& e) { e.second.count++; h->insert(e); }
};

/************************ ResetCounts **************************/

//Function object to perform an operation during traversal
struct ResetCounts : public unary_function<pItemset&, void>
{
    int p;
    ResetCounts(int partition) : p(partition) { }
    void operator() (pItemset& e) {e.second.count=0; e.second.partition=p;}
};

/************************* MoveLarge **************************/

//Function object to perform an operation during traversal
struct MoveLarge : public unary_function<pItemset&, void>
{
    pHashtree *L;
    int minCount;
    MoveLarge(pHashtree &h, int m)
    	{ L = &h; minCount = m; }
    void operator() (pItemset& e)
    {
	if (e.second.count >= minCount)
	    L->move(e);
    }
};

/************************* MoveLargep ***********************/
//Function object to perform an operation during traversal
struct MoveLargep : public unary_function<pItemset&, void>
{
    pHashtree *L;
    double minsupp;
    int psize, rows, partition;

    MoveLargep(pHashtree &h, double m, int p, int row, int ps)
    	{ L = &h; minsupp = m; rows = row; psize = ps; partition = p; }
    void operator() (pItemset& e)
    {
	if (e.second.count/(double)(rows+(partition-e.second.partition)
		*psize) >= minsupp)
	    L->move(e);
    }
};

/************************* MoveLargep2 ***********************/
//Function object to perform an operation during traversal
struct MoveLargep2 : public unary_function<pItemset&, void>
{
    pHashtree *newTree;
    double minsupp;
    int psize, lastpartSize, partition, lastpart;

    MoveLargep2(int p,pHashtree& h1,double m,int row,int ps,
	    int noparts)
    	{ newTree = &h1; minsupp = m; lastpartSize = row; psize
	    = ps; partition = p; lastpart = noparts; }
    void operator() (pItemset& e)
    {
	if (e.second.count/(double)(lastpartSize+(lastpart+partition-
	    	 e.second.partition)*psize) >= minsupp)
	    newTree->move(e);
    }
};

/************************* MoveLargep3 ***********************/
//Function object to perform an operation during traversal
struct MoveLargep3 : public unary_function<pItemset&, void>
{
    pHashtree *L;
    double minsupp;
    int rows;

    MoveLargep3(pHashtree &h, double m, int row)
    	{ L = &h; minsupp = m; rows = row; }
    void operator() (pItemset& e)
    {
	if (e.second.count+e.second.partition/(double)rows >= minsupp ||
		e.first.size() == 1)
	    L->move(e);
    }
};

/************************* CopyPartition ***********************/
//Function object to perform an operation during traversal
struct CopyPartition : public unary_function<pItemset&, void>
{
    pHashtree *tempTree;
    int partition;

    CopyPartition(int p,pHashtree& h1)
    	{ tempTree = &h1; partition = p; }
    void operator() (pItemset& e)
    {
	if (e.second.partition == partition)
	    tempTree->insert(e);
    }
};

/************************* InsertLarge **************************/

//Function object to perform an operation during traversal
struct InsertLarge : public unary_function<pItemset&, void>
{
    pHashtree *L;
    double minsupp;
    int rows;
    int partition;
    int psize;
    InsertLarge(pHashtree &h, double Minsupp, int Rows, int Partition,
	    int Psize) :
	    L(&h), minsupp(Minsupp), rows(Rows), partition(Partition),
	    psize(Psize) { }
    void operator() (pItemset& e)
    {
	if (e.second.count/(double)(rows+(partition-e.second.partition)*psize) >= minsupp)
	    L->insert(e);
    }
};

/************************* InsertSmall **************************/

//Function object to perform an operation during traversal
struct InsertSmall : public unary_function<pItemset&, void>
{
    pHashtree *S;
    double minsupp;
    int rows;
    int partition;
    int psize;
    InsertSmall(pHashtree &h, double Minsupp, int Rows, int Partition,
	    int Psize) :
	    S(&h), minsupp(Minsupp), rows(Rows), partition(Partition),
	    psize(Psize) { }
    void operator() (pItemset& e)
    {
	if (e.second.count/(double)(rows+(partition-e.second.partition)*psize) < minsupp)
	    S->insert(e);
    }
};

/********************** MoveLargeSmall ***********************/

//Function object to perform an operation during traversal
struct MoveLargeSmall : public unary_function<pItemset&, void>
{
    pHashtree *L;
    pHashtree *N;
    int minCount;
    MoveLargeSmall(pHashtree& h1, pHashtree& h2, int m) :
	    L(&h1), N(&h2), minCount(m) { }
    void operator() (pItemset& e)
    {
	if (e.second.count >= minCount)
	    L->move(e);
	else
	    N->move(e);
    }
};

/************************* Move **************************/

//Function object to perform an operation during traversal
struct Move : public unary_function<pItemset&, void>
{
    pItemsetBag *B;
    Move(pItemsetBag &b) { B = &b; }
    void operator() (pItemset& e)
    {
	B->push_back(make_pair(Itemset(),e.second));
	(B->back()).first.swap(e.first);
    }
};

/*********************** InsertBag ************************/

//Function object to perform an operation during traversal
struct InsertBag : public unary_function<pItemset&, void>
{
    pItemsetBag& B;
    InsertBag(pItemsetBag &b) : B(b) {}
    void operator() (pItemset& e)
    {
	B.push_back(e);
    }
};

/************************* Insert **************************/

//Function object to perform an operation during traversal
struct Insert : public unary_function<pItemset&, void>
{
    pHashtree *h;
    Insert(pHashtree &H) : h(&H) { }
    void operator() (pItemset& e) { h->insert(e); }
};

/************************* Remove **************************/

//Function object to perform an operation during traversal
struct Remove : public unary_function<pItemset&, void>
{
    pHashtree *h;
    Remove(pHashtree &H) : h(&H) { }
    void operator() (pItemset& e) { h->remove(e.first); }
};

/********************** Move into a tree ********************/

//Function object to perform an operation during traversal
struct MoveTree : public unary_function<pItemset&, void>
{
    pHashtree *h;
    MoveTree(pHashtree &H) : h(&H) { }
    void operator() (pItemset& e) { h->move(e); }
};

/********************** NegBorderIncrement() *******************/

struct Extend : public unary_function<pItemset&, void>
{
    Itemset *i1;
    const pHashtree *h;
    pHashtree *result;
    int partition;
    Extend(Itemset& I,const pHashtree* H,pHashtree *Result,int p) :
	    i1(&I), h(H), result(Result), partition(p) { }
    void operator() (const pItemset& e)
    {
	bool shift = false;
	Itemset::const_iterator i, item=0;
	Itemset::const_iterator j = e.first.begin();
	for (i=i1->begin(); i < i1->end()-1; i++ )
	{
	    if ( shift==false && *j < *i ) {
		item = j++;
		shift=true;
	    }
	    if ( *j != *i ) //itemsets differ in more than 1 position
		return;
	    j++;
	}
	if ( shift==false ) {
	    if ( *i == *j ) //itemsets are exactly identical
		return;
	    item = j;
	}
    //item points to the item in e.first which is not present in *i1
    //*i1 and e.first differ in exactly 1 position

    //----- make a candidate out of *i1 and e.first
	Itemset candidate;
	candidate.reserve(i1->size()+1);
	candidate.insert(candidate.begin(),i1->begin(),i1->end());
	candidate.push_back(*item);

	//prune
	if ( h->contains(candidate) )
	    return;
	if ( i1->size() > 1 ) {
	    Itemset::iterator k;
	    for ( k=candidate.begin(); k<candidate.end()-1; k++ ) {
		Item temp = *k;
		candidate.erase(k);
		if ( ! h->contains(candidate) )
		    return;
		candidate.insert(k,temp);
	    }
	}

	pItemset newEntry;
	newEntry.first.swap(candidate);
	newEntry.second.count = 0;
	newEntry.second.partition = partition;
	result->move(newEntry);
    }
};

struct GetCandidates : public unary_function<pItemset&, void>
{
    const pHashtree *h;
    pHashtree *result;
    int partition;
    GetCandidates(const pHashtree& H, pHashtree &Result, int p) :
	    h(&H), result(&Result), partition(p) { }
    void operator ()(pItemset& e)
    	{ for_each(*h,Extend(e.first,h,result,partition),e.first.size()); }
};

void NegBorderIncrement1(pHashtree& L, pHashtree& N, pHashtree& C,
	double minsupp, int rows, int partition, int psize)
{
/* N contained negative border of L, the set of large itemsets. The
   counts of L and N have been updated somehow. The minimum count of a
   large itemset is minCount. This function moves the itemsets from L
   which have become small to N and vice versa. It then creates new
   candidates, i.e. itemsets which should now be in N, but were neither
   in N nor in L previously. These new candidates are put into C. */

    pHashtree moveNL, moveLN;
    for_each(N,InsertLarge(moveNL,minsupp,rows,partition,psize));
    for_each(moveNL,Remove(N));
    for_each(moveNL,Insert(L));
    for_each(L,InsertSmall(moveLN,minsupp,rows,partition,psize));
    for_each(moveLN,Remove(L));
    for_each(moveLN,MoveTree(N));
    moveLN.clear();
    for_each(moveNL,GetCandidates(L,C,partition));
}

void NegBorderIncrement2(pHashtree& L, pHashtree& N, pHashtree& C,
	int minCount, int partition)
{
/* N contained negative border of L, the set of large itemsets. C
   contains candidates whose counts have just been updated somehow. The
   minimum count of a large itemset is minCount. This function moves
   large itemsets from C to L and small ones to N. It then creates new
   candidates, i.e. itemsets which should now be in N, but were neither
   in N nor in L nor in C previously. These new candidates are put into
   C after making it empty. */

    pHashtree moveNL;
    for_each(C,MoveLargeSmall(moveNL,N,minCount));
    for_each(moveNL,Insert(L));
    C.clear();
    for_each(moveNL,GetCandidates(L,C,partition));
}

#endif
