/**********************************************************************
    Copyright (C) 2004 Database Systems Lab, Supercomputer Education and
    Research Centre, Indian Institute of Science, Bangalore, INDIA.
    http://dsl.serc.iisc.ernet.in

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***********************************************************************/


/***********************************************************************
 AUTHOR: Vikram Pudi

 DESCRIPTION:

***********************************************************************/

#ifndef ITEMSETDAG_H_
#define ITEMSETDAG_H_

#include "item.h" //this includes vector.h
#include "list.h"
#include <math.h> //for ceil()

/***************************** DAG-node *******************************/
struct DagNode
{ //each DagNode corresponds to an itemset

    list<DagNode*> llist;
    list<DagNode*> rlist;

    DagNode* lchild;
    list<DagNode*>::iterator lback; //back pointer
    int litem;

    DagNode* rchild;
    list<DagNode*>::iterator rback; //back pointer
    int ritem;

    int partial_count;
    int stimulus; //a value between 0 and 2, inclusive
    long ftid; //first TID: when this itemset was inserted
    long ltid; //last TID: when this itemset was last seen

    bool small(long tid, double minsupp) const
    	{ return (partial_count < (int)ceil(minsupp*(tid-ftid+1))); }
    bool large(long tid, double minsupp) const
    	{ return (partial_count >= (int)ceil(minsupp*(tid-ftid+1))); }

    void clear();
    int erase_with_supersets(bool right);
    int trigger(long tid, double minsupp);
    void incrCount(long tid);
    void cleanup()
    {
	partial_count = 0;
	stimulus = 0;
	ftid = ltid = 0;

	list<DagNode*>::iterator i;
	for (i = llist.begin(); i != llist.end(); i++)
	    (*i)->cleanup();
    }

//----------------- i/o operations ---------------
    void output(ostream&, Itemset&, bool tid, long ltid, double minsupp)
	    const;
};

/*************************** Itemset-DAG ******************************/
class ItemsetDag
{
    vector<DagNode> singletons;
    size_t no_nodes;

public:

//--------------- construct/destroy --------------
    ItemsetDag() { }
    ItemsetDag(size_t s) { setNoItems(s); }
    ~ItemsetDag() { clear(); }

    void clear()
    {
	vector<DagNode>::iterator i;
	for (i = singletons.begin(); i != singletons.end(); i++)
	    i->clear();
	no_nodes = 0;
    }

//------------ functions to access members -------
    void setNoItems(size_t s)
    {
	singletons.resize(s);
	no_nodes = s;
	for (size_t i = 0; i < singletons.size(); i++)
	{
	    singletons[i].litem = i;
	    singletons[i].lchild = singletons[i].rchild = 0;
	    singletons[i].partial_count = 0;
	    singletons[i].stimulus = 0;
	    singletons[i].ftid = singletons[i].ltid = 0;
	}
    }

    size_t noItems() const { return singletons.size(); }
    size_t size() const { return no_nodes; }

//----------------- misc. functions --------------
    void swap(ItemsetDag& d) //swap contents of 2 ItemsetDags
    {
	size_t tno_nodes = no_nodes;
	no_nodes = d.no_nodes;
	d.no_nodes =  tno_nodes;
	singletons.swap(d.singletons);
    }

//----------------- i/o operations ---------------
    void input(istream& s, bool tid, long ftid, long ltid);
    void output(ostream&, bool tid, long ltid, double minsupp) const;
    DagNode* insert(Itemset&, int ftid, int ltid, int count);

//------------------ crunch tuples ---------------
    void crunch(const Itemset& I, long tid, double minsupp);
    void incrCount(const Itemset& I, long tid);
    void cleanup()
    {
	vector<DagNode>::iterator i;
	for (i = singletons.begin(); i != singletons.end(); i++)
	    i->cleanup();
    }
};

inline ostream& operator<<(ostream& s, const ItemsetDag& d)
{
    d.output(s, true, 0, 1);
    return s;
}

inline istream& operator>>(istream& s, ItemsetDag& d)
{
    d.input(s, true, 0, 0);
    return s;
}

#endif
