/**********************************************************************
    Copyright (C) 2004 Database Systems Lab, Supercomputer Education and
    Research Centre, Indian Institute of Science, Bangalore, INDIA.
    http://dsl.serc.iisc.ernet.in

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***********************************************************************/


/********************* HashTree<Key,T> Traversals **********************
 Traversal routines on hashtrees.
 
 We use the functor concept of C++ and define multiple types of
 traversal routines.  We have routines to traverse hashtrees -- either
 all itemsets in a hashtree or subsets of a given itemset or supersets
 of it.  This file is to be included in hashtree.h and has been put in a
 separate file in order to reduce bulk.  To see good examples of how to
 use these routines, refer to the apriori.h file.

 Leaf node traversals are defined first followed by internal node
 traversals.  These will not be used outside of this file and are
 defined only to be used by the actual hashtree traversals that are
 defined next.  Within each of the above sets of traversals, we have --
 
    for_each, for_each_subset and for_each_extension

 An extra integer argument can be passed to these routines to specify
 the length of itemsets that must be traversed.  If no length is given,
 all itemsets are traversed.
 
 To refer to the exact syntax of each functor, see the cp. code for it
 below.  Every alternate function is same as the previous, except that
 it is a const version.
***********************************************************************/
#ifndef HASHTRAVERSE_H_
#define HASHTRAVERSE_H_
#include <bvector.h>

//---------- Leaf node traversals ------------
//------- for each itemset do f -----
template <class Key,class T,class Function>
inline Function for_each(Leaf_node<Key,T>& h, Function f)
{
    typedef Leaf_node<Key,T> leaf_type;
    leaf_type::iterator j;
    for (j = h.begin(); j != h.end(); j++)
	f(*j);
    return f;
}

template <class Key,class T,class Function>
inline Function for_each(const Leaf_node<Key,T>& h, Function f)
{
    typedef Leaf_node<Key,T> leaf_type;
    leaf_type::const_iterator j;
    for (j = h.begin(); j != h.end(); j++)
	f(*j);
    return f;
}

//------- for each itemset of size n do f -----
template <class Key,class T,class Function>
inline Function for_each(Leaf_node<Key,T>& h, Function f, int n)
{
    typedef Leaf_node<Key,T> leaf_type;
    leaf_type::iterator j;
    for (j = h.begin(); j != h.end(); j++)
	if ((int)(j->first).size() == n)
	    f(*j);
    return f;
}

template <class Key,class T,class Function>
inline Function for_each(const Leaf_node<Key,T>& h, Function f, int n)
{
    typedef Leaf_node<Key,T> leaf_type;
    leaf_type::const_iterator j;
    for (j = h.begin(); j != h.end(); j++)
	if ((int)(j->first).size() == n)
	    f(*j);
    return f;
}

//------- for each subset of itemset i do f -----
template <class Key,class T,class Function>
inline Function for_each_subset(Leaf_node<Key,T>& h, Function f,
	const bit_vector& i)
{
    typedef Leaf_node<Key,T> leaf_type;
    leaf_type::iterator j;
    for (j = h.begin(); j != h.end(); j++)
    {
	typename Key::iterator k;
	for (k=j->first.begin();k!=j->first.end();k++)
	    if (i[k->id]==false) //j is not a subset of i
		break;
	if (k==j->first.end()) //j is a subset of i
	    f(*j);
    }
    return f;
}

template <class Key,class T,class Function>
inline Function for_each_subset(const Leaf_node<Key,T>& h, Function f,
	const bit_vector& i)
{
    typedef Leaf_node<Key,T> leaf_type;
    leaf_type::const_iterator j;
    for (j = h.begin(); j != h.end(); j++)
    {
	typename Key::const_iterator k;
	for (k=j->first.begin();k!=j->first.end();k++)
	    if (i[k->id]==false) //j is not a subset of i
		break;
	if (k==j->first.end()) //j is a subset of i
	    f(*j);
    }
    return f;
}

//------- for each subset of itemset i of size n do f -----
template <class Key,class T,class Function>
inline Function for_each_subset(Leaf_node<Key,T>& h, Function f, const
	bit_vector& i, int n)
{
    typedef Leaf_node<Key,T> leaf_type;
    leaf_type::iterator j;
    for (j = h.begin(); j != h.end(); j++)
    {
	if ((int)j->first.size() == n) {
	    typename Key::iterator k;
	    for (k=j->first.begin();k!=j->first.end();k++)
		if (i[k->id]==false) //j is not a subset of i
		    break;
	    if (k==j->first.end()) //j is a subset of i
		f(*j);
	}
    }
    return f;
}

template <class Key,class T,class Function>
inline Function for_each_subset(const Leaf_node<Key,T>& h, Function f, const
	bit_vector& i, int n)
{
    typedef Leaf_node<Key,T> leaf_type;
    leaf_type::const_iterator j;
    for (j = h.begin(); j != h.end(); j++)
    {
	if ((int)j->first.size() == n) {
	    typename Key::const_iterator k;
	    for (k=j->first.begin();k!=j->first.end();k++)
		if (i[k->id]==false) //j is not a subset of i
		    break;
	    if (k==j->first.end()) //j is a subset of i
		f(*j);
	}
    }
    return f;
}

//------- for each extension of itemset i do f -----
template <class Key,class T,class Function>
inline Function for_each_extension(Leaf_node<Key,T>& h, Function f, const Key& i)
{ //utilizes fact that itemsets are in lexicographic order
    typedef Leaf_node<Key,T> leaf_type;
    leaf_type::iterator pos;
    pos = lower_bound(h.begin(),h.end(),i,less_pair<Key,T>());
	    //binary search for i
    while (pos != h.end() && includes(pos->first,i))
	f(*pos++);
    return f;
}

template <class Key,class T,class Function>
inline Function for_each_extension(const Leaf_node<Key,T>& h, Function f, const Key& i)
{ //utilizes fact that itemsets are in lexicographic order
    typedef Leaf_node<Key,T> leaf_type;
    leaf_type::const_iterator pos;
    pos = lower_bound(h.begin(),h.end(),i,less_pair<Key,T>());
	    //binary search for i
    while (pos != h.end() && includes(pos->first,i))
	f(*pos++);
    return f;
}

//------- for each extension of itemset i of size n do f -----
template <class Key,class T,class Function>
inline Function for_each_extension(Leaf_node<Key,T>& h, Function f, const Key& i, int n)
{ //utilizes fact that itemsets are in lexicographic order
    typedef Leaf_node<Key,T> leaf_type;
    leaf_type::iterator pos;
    pos = lower_bound(h.begin(),h.end(),i,less_pair<Key,T>());
	    //binary search for i
    while (pos != h.end() && includes(pos->first,i)) {
	if ((int)pos->first.size()==n)
	    f(*pos);
	pos++;
    }

    return f;
}

template <class Key,class T,class Function>
inline Function for_each_extension(const Leaf_node<Key,T>& h, Function f, const Key& i, int n)
{ //utilizes fact that itemsets are in lexicographic order
    typedef Leaf_node<Key,T> leaf_type;
    leaf_type::const_iterator pos;
    pos = lower_bound(h.begin(),h.end(),i,less_pair<Key,T>());
	    //binary search for i
    while (pos != h.end() && includes(pos->first,i)) {
	if ((int)pos->first.size()==n)
	    f(*pos);
	pos++;
    }

    return f;
}

//---------- Internal node traversals ------------
//------- for each itemset do f -----
template <class Key,class T,class Function>
Function for_each(Internal_node<Key,T>& h, Function f)
{
    Internal_node<Key,T>::iterator j;
    for (j = h.begin(); j != h.end(); j++)
	if (*j != 0)
        {
	    if ((*j)->leaf())
		for_each((*((Leaf_node<Key,T>*)(*j))),f);
	    else
		for_each((*((Internal_node<Key,T>*)(*j))),f);
	}
    return f;
}

template <class Key,class T,class Function>
Function for_each(const Internal_node<Key,T>& h, Function f)
{
    Internal_node<Key,T>::const_iterator j;
    for (j = h.begin(); j != h.end(); j++)
	if (*j != 0)
        {
	    if ((*j)->leaf())
		for_each((*((const Leaf_node<Key,T>*)(*j))),f);
	    else
		for_each((*((const Internal_node<Key,T>*)(*j))),f);
	}
    return f;
}

//------- for each itemset of size n do f -----
template <class Key,class T,class Function>
Function for_each(Internal_node<Key,T>& h, Function f, int n, int level)
{
    if (level > n)
	return f;
    if (level == n) {
	if (h.front() != 0)
	    return (for_each((*((Leaf_node<Key,T>*)(h.front()))),f));
	else
	    return f;
    }

    Internal_node<Key,T>::iterator j;
    for (j = h.begin()+1; j < h.end(); j++)
	if (*j != 0)
        {
	    if ((*j)->leaf())
		for_each((*((Leaf_node<Key,T>*)(*j))),f,n);
	    else
		for_each((*((Internal_node<Key,T>*)(*j))),f,n,level+1);
	}
    return f;
}

template <class Key,class T,class Function>
Function for_each(const Internal_node<Key,T>& h, Function f, int n, int level)
{
    if (level > n)
	return f;
    if (level == n) {
	if (h.front() != 0)
	    return (for_each((*((Leaf_node<Key,T>*)(h.front()))),f));
	else
	    return f;
    }

    Internal_node<Key,T>::const_iterator j;
    for (j = h.begin()+1; j < h.end(); j++)
	if (*j != 0)
        {
	    if ((*j)->leaf())
		for_each((*((const Leaf_node<Key,T>*)(*j))),f,n);
	    else
		for_each((*((const Internal_node<Key,T>*)(*j))),f,n,level+1);
	}
    return f;
}

//------- for each subset of itemset i do f -----
template <class Key,class T,class Function>
Function for_each_subset(Internal_node<Key,T>& h, Function f, const Key& i,
	const bit_vector& iset,	int level)
{
    if (h[0] != 0)
	for_each_subset((*((Leaf_node<Key,T>*)(h[0]))),f,iset);
    int n = (int)i.size();
    if (!(level < n))
	return f;

    bit_vector visited(h.size(),false);
    for (int k = level; k < (int)i.size(); k++)
    {
	int index = h.hashFunc(i[k]);

	if ((!(visited[index])) && (h[index] != 0))
        {
	    visited[index]=true;
	    if ((h[index])->leaf())
		for_each_subset((*((Leaf_node<Key,T>*)(h[index]))),f,iset);
	    else
		for_each_subset((*((Internal_node<Key,T>*)(h[index]))),f,i,
			iset,level+1);
	}
    }
    return f;
}

template <class Key,class T,class Function>
Function for_each_subset(const Internal_node<Key,T>& h, Function f, const
	Key& i, const bit_vector& iset, int level)
{
    if (h[0] != 0)
	for_each_subset((*((const Leaf_node<Key,T>*)(h[0]))),f,iset);
    int n = (int)i.size();
    if (!(level < n))
	return f;

    bit_vector visited(h.size(),false);
    for (int k = level; k < (int)i.size(); k++)
    {
	int index = h.hashFunc(i[k]);

	if ((!(visited[index])) && (h[index] != 0))
        {
	    visited[index]=true;
	    if ((h[index])->leaf())
		for_each_subset((*((const Leaf_node<Key,T>*)(h[index]))),f,iset);
	    else
		for_each_subset((*((const Internal_node<Key,T>*)(h[index]))),f,i,
			iset,level+1);
	}
    }
    return f;
}

//------- for each subset of itemset i of size n do f -----
template <class Key,class T,class Function>
Function for_each_subset(Internal_node<Key,T>& h, Function f, const Key& i,
	const bit_vector& iset, int n, int level)
{
    if (level == n && h[0] != 0)
	for_each_subset((*((const Leaf_node<Key,T>*)(h[0]))),f,iset);
    if (!(level < n))
	 return f;

    bit_vector visited(h.size(),false);
    for (int k = level; k < (int)i.size(); k++)
    {
	int index = h.hashFunc(i[k]);

	if ((!(visited[index])) && (h[index] != 0))
        {
	    visited[index]=true;
	    if ((h[index])->leaf())
		for_each_subset((*((Leaf_node<Key,T>*)(h[index]))),f,iset,n);
	    else
		for_each_subset((*((Internal_node<Key,T>*)(h[index]))),f,i,
			iset,n,level+1);
	}
    }
    return f;
}

template <class Key,class T,class Function>
Function for_each_subset(const Internal_node<Key,T>& h, Function f, const Key& i,
	const bit_vector& iset, int n, int level)
{
    if (level == n && h[0] != 0)
	for_each_subset((*((const Leaf_node<Key,T>*)(h[0]))),f,iset);
    if (!(level < n))
	 return f;

    bit_vector visited(h.size(),false);
    for (int k = level; k < (int)i.size(); k++)
    {
	int index = h.hashFunc(i[k]);

	if ((!(visited[index])) && (h[index] != 0))
        {
	    visited[index]=true;
	    if ((h[index])->leaf())
		for_each_subset((*((const Leaf_node<Key,T>*)(h[index]))),f,iset,n);
	    else
		for_each_subset((*((const Internal_node<Key,T>*)(h[index]))),f,i,
			iset,n,level+1);
	}
    }
    return f;
}

//------- for each extension of itemset i do f -----
template <class Key,class T,class Function>
Function for_each_extension(Internal_node<Key,T>& h, Function f, const Key& i)
{
    Internal_node<Key,T>::iterator j;
    for (j = h.begin(); j != h.end(); j++)
	if (*j != 0)
        {
	    if ((*j)->leaf())
		for_each_extension((*((Leaf_node<Key,T>*)(*j))),f,i);
	    else
		for_each_extension((*((Internal_node<Key,T>*)(*j))),f,i);
	}
    return f;
}

template <class Key,class T,class Function>
Function for_each_extension(const Internal_node<Key,T>& h, Function f, const
	Key& i, int level)
{
    Internal_node<Key,T>::const_iterator j;
    for (j = h.begin(); j != h.end(); j++)
	if (*j != 0)
        {
	    if ((*j)->leaf())
		for_each_extension((*((const Leaf_node<Key,T>*)(*j))),f,i);
	    else
		for_each_extension((*((const Internal_node<Key,T>*)(*j))),f,i);
	}
    return f;
}

//------- for each extension of itemset i of size n do f -----
template <class Key,class T,class Function>
Function for_each_extension(Internal_node<Key,T>& h, Function f, const Key& i,
	int n, int level)
{
    if (level > n)
	return f;

    Internal_node<Key,T>::iterator j;
    if (level == n)
    {
	if ((j = h.begin()) != 0)
	    for_each_extension((*((Leaf_node<Key,T>*)(*j))),f,i,n);
	return f;
    }
    for (j = h.begin()+1; j < h.end(); j++)
	if (*j != 0)
        {
	    if ((*j)->leaf())
		for_each_extension((*((Leaf_node<Key,T>*)(*j))),f,i,n);
	    else
		for_each_extension((*((Internal_node<Key,T>*)(*j))),f,i,n,level+1);
	}
    return f;
}

template <class Key,class T,class Function>
Function for_each_extension(const Internal_node<Key,T>& h, Function f, const Key& i,
	int n, int level)
{
    if (level > n)
	return f;

    Internal_node<Key,T>::const_iterator j;
    if (level == n)
    {
	if ((j = h.begin()) != 0)
	    for_each_extension((*((const Leaf_node<Key,T>*)(*j))),f,i,n);
	return f;
    }
    for (j = h.begin()+1; j < h.end(); j++)
	if (*j != 0)
        {
	    if ((*j)->leaf())
		for_each_extension((*((const Leaf_node<Key,T>*)(*j))),f,i,n);
	    else
		for_each_extension((*((const Internal_node<Key,T>*)(*j))),f,i,n,level+1);
	}
    return f;
}

//---------- Actual tree traversals ------------
//------- for each itemset do f -----
template <class Key,class T,class Function>
inline Function for_each(HashTree<Key,T>& h, Function f)
{
    if (h.getRoot() == 0)
	return f;
    return (((h.getRoot())->leaf())?
	(for_each((*((Leaf_node<Key,T>*)h.getRoot())),f)) :
	(for_each((*((Internal_node<Key,T>*)h.getRoot())),f)));
}

template <class Key,class T,class Function>
inline Function for_each(const HashTree<Key,T>& h, Function f)
{
    if (h.getRoot() == 0)
	return f;
    return (((h.getRoot())->leaf())?
	(for_each((*((const Leaf_node<Key,T>*)h.getRoot())),f)) :
	(for_each((*((const Internal_node<Key,T>*)h.getRoot())),f)));
}

//------- for each itemset of size n do f -----
template <class Key,class T,class Function>
inline Function for_each(HashTree<Key,T>& h, Function f, int n)
{
    if (h.getRoot() == 0)
	return f;
    return (((h.getRoot())->leaf())?
	(for_each((*((Leaf_node<Key,T>*)h.getRoot())),f,n)) :
	(for_each((*((Internal_node<Key,T>*)h.getRoot())),f,n,0)));
}

template <class Key,class T,class Function>
inline Function for_each(const HashTree<Key,T>& h, Function f, int n)
{
    if (h.getRoot() == 0)
	return f;
    return (((h.getRoot())->leaf())?
	(for_each((*((const Leaf_node<Key,T>*)h.getRoot())),f,n)) :
	(for_each((*((const Internal_node<Key,T>*)h.getRoot())),f,n,0)));
}

//------- for each subset of itemset i do f -----
template <class Key,class T,class Function>
inline Function for_each_subset(HashTree<Key,T>& h, Function f, const Key& i)
{
    if (h.getRoot() == 0)
	return f;
    static bit_vector iset(h.noItems(),false);
    for (typename Key::const_iterator j=i.begin(); j!=i.end(); j++)
	iset[j->id] = true;
    Function retval = (((h.getRoot())->leaf())?
	(for_each_subset((*((Leaf_node<Key,T>*)h.getRoot())),f,iset)) :
	(for_each_subset((*((Internal_node<Key,T>*)h.getRoot())),f,i,iset,0)));
    for (typename Key::const_iterator j=i.begin(); j!=i.end(); j++)
	iset[j->id] = false;
    return retval;
}

template <class Key,class T,class Function>
inline Function for_each_subset(const HashTree<Key,T>& h, Function f, const Key& i)
{
    if (h.getRoot() == 0)
	return f;
    static bit_vector iset(h.noItems(),false);
    for (typename Key::const_iterator j=i.begin(); j!=i.end(); j++)
	iset[j->id] = true;
    Function retval = (((h.getRoot())->leaf())?
	(for_each_subset((*((const Leaf_node<Key,T>*)h.getRoot())),f,iset)) :
	(for_each_subset((*((const Internal_node<Key,T>*)h.getRoot())),f,i,iset,0)));
    for (typename Key::const_iterator j=i.begin(); j!=i.end(); j++)
	iset[j->id] = false;
    return retval;
}

//------- for each subset of itemset i of size n do f -----
template <class Key,class T,class Function>
inline Function for_each_subset(HashTree<Key,T>& h, Function f, const Key& i, int n)
{
    if ((int)i.size() < n)
	return f;
    if (h.getRoot() == 0)
	return f;
    static bit_vector iset(h.noItems(),false);
    for (typename Key::const_iterator j=i.begin(); j!=i.end(); j++)
	iset[j->id] = true;
    Function retval = (((h.getRoot())->leaf())?
	(for_each_subset((*((Leaf_node<Key,T>*)h.getRoot())),f,iset,n)) :
	(for_each_subset((*((Internal_node<Key,T>*)h.getRoot())),f,i,iset,n,0)));
    for (typename Key::const_iterator j=i.begin(); j!=i.end(); j++)
	iset[j->id] = false;
    return retval;
}

template <class Key,class T,class Function>
inline Function for_each_subset(const HashTree<Key,T>& h, Function f, const Key& i, int n)
{
    if ((int)i.size() < n)
	return f;
    if (h.getRoot() == 0)
	return f;
    static bit_vector iset(h.noItems(),false);
    for (typename Key::const_iterator j=i.begin(); j!=i.end(); j++)
	iset[j->id] = true;
    Function retval = (((h.getRoot())->leaf())?
	(for_each_subset((*((const Leaf_node<Key,T>*)h.getRoot())),f,iset,n)) :
	(for_each_subset((*((const Internal_node<Key,T>*)h.getRoot())),f,i,iset,n,0)));
    for (typename Key::const_iterator j=i.begin(); j!=i.end(); j++)
	iset[j->id] = false;
    return retval;
}

//------- for each extension of itemset i do f -----
template <class Key,class T,class Function>
inline Function for_each_extension(HashTree<Key,T>& h, Function f, const Key& i)
{
    Hashnode *child = h.getRoot();
    int level = 0; //level of child; root is at level 0
    while (child != 0) //internal node
    {
	if (child->leaf())
	    return (for_each_extension((*((Leaf_node<Key,T>*)child)),f,i));
	Internal_node<Key,T> *node=(Internal_node<Key,T>*)(child);
	if (level >= (int)i.size())
	    return (for_each_extension(*node,f,i));
	size_t index = h.hashFunc(i[level]);    
	child = (*node)[index];
	level++;
    }
}

template <class Key,class T,class Function>
inline Function for_each_extension(const HashTree<Key,T>& h, Function f, const Key& i)
{
    const Hashnode *child = h.getRoot();
    int level = 0; //level of child; root is at level 0
    while (child != 0) //internal node
    {
	if (child->leaf())
	    return (for_each_extension((*((const Leaf_node<Key,T>*)child)),f,i));
	const Internal_node<Key,T> *node=(const Internal_node<Key,T>*)(child);
	if (level >= (int)i.size())
	    return (for_each_extension(*node,f,i));
	size_t index = h.hashFunc(i[level]);    
	child = (*node)[index];
	level++;
    }
}

//------- for each extension of itemset i of size n do f -----
template <class Key,class T,class Function>
inline Function for_each_extension(HashTree<Key,T>& h, Function f, const Key& i, int n)
{
    if (n < (int)i.size())
	return f;
    Hashnode *child = h.getRoot();
    int level = 0; //level of child; root is at level 0
    while (child !=0) //internal node
    {
	if (child->leaf())
	    return (for_each_extension((*((Leaf_node<Key,T>*)child)),f,i,n));
	Internal_node<Key,T> *node=(Internal_node<Key,T>*)(child);
	if (level >= (int)i.size())
	    return (for_each_extension(*node,f,i,n,level));
	size_t index = h.hashFunc(i[level]);    
	child = (*node)[index];
	level++;
    }
    return f;
}

template <class Key,class T,class Function>
inline Function for_each_extension(const HashTree<Key,T>& h, Function f, const Key& i, int n)
{
    if (n < (int)i.size())
	return f;
    const Hashnode *child = h.getRoot();
    int level = 0; //level of child; root is at level 0
    while (child !=0) //internal node
    {
	if (child->leaf())
	    return (for_each_extension((*((const Leaf_node<Key,T>*)child)),f,i,n));
	const Internal_node<Key,T> *node=(const Internal_node<Key,T>*)(child);
	if (level >= (int)i.size())
	    return (for_each_extension(*node,f,i,n,level));
	size_t index = h.hashFunc(i[level]);    
	child = (*node)[index];
	level++;
    }
    return f;
}

#endif
