/**********************************************************************
    Copyright (C) 2004 Database Systems Lab, Supercomputer Education and
    Research Centre, Indian Institute of Science, Bangalore, INDIA.
    http://dsl.serc.iisc.ernet.in

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***********************************************************************/


/***********************************************************************
 The AprioriGen Function and other functions useful for Apriori like
 algorithms.
 
 We use the functor concept of C++ and multiple types of traversal
 routines to accomplish the tasks.  We have routines to traverse
 hashtrees -- either all itemsets in a hashtree or subsets of a given
 itemset or supersets of it.  These traversal routines are defined in
 hashTraverse.h which is included from hashtree.h.

 As expected, to define a functor, we need to define all the functions
 and functors that it uses, before the actual code of this functor. 
 However we think in a top-down approach here.  Therefore to understand
 the code of each functor, read each section from bottom up.
***********************************************************************/
#ifndef APRIORI_H_
#define APRIORI_H_

#include "cItemset.h"
#include "taxonomy.h"


/*#include "mat.h"
 * #include "inverse.h"
 */
#include "my_matrix.h"
//#include "../matrix.h"

#define DEBUG 0
/************************ AprioriGen *****************************/

//Function object to perform an operation during traversal
struct Apriori_gen2 : public unary_function<const cItemset&, void>
{
    const cHashtree *L;
    cHashtree *C;
    const Itemset *i1;
    static long no_pruned;

    Apriori_gen2(const cHashtree *l, cHashtree *c,const Itemset *i)
	    { L=l; C=c; i1=i; }

    void operator() (const cItemset& e)
    {
	/* preconditions:
	   1) e.first which is an itemset, must have size>=1
	   2) e.first.size() == i1->size() is true */

	const Itemset& i2 = e.first;
	if (equal(i1->begin(),(i1->end())-1,i2.begin()) &&
		i1->back() < i2.back())
	{
	    Itemset candidate;
	    candidate.reserve(i1->size()+1);
	    candidate.insert(candidate.begin(),i1->begin(),i1->end());
	    candidate.push_back(i2.back());

	    //prune
	    if ( i1->size() > 1 ) {
		Itemset::iterator j;
		for ( j=candidate.begin(); j<candidate.end()-2; j++ ) {
		    Item temp = *j;
		    candidate.erase(j);
		    if ( ! L->contains(candidate) )
		    {
			no_pruned++;
			return;
		    }
		    candidate.insert(j,temp);
		}
	    }

	    cItemset newEntry;
	    newEntry.first.swap(candidate);
	    newEntry.second = 0;
	    C->move(newEntry);
	}
    }
};

//Function object to perform an operation during traversal
struct Apriori_gen1 : public unary_function<const cItemset&, void>
{
    const cHashtree *L;
    cHashtree *C;

    Apriori_gen1(const cHashtree *l, cHashtree *c)
	    { L=l; C=c; }

    void operator() (const cItemset& e)
    {
	//precondition: e.first must have size >= 1

	Itemset i(e.first.begin(), e.first.end()-1);
	for_each_extension(*L, Apriori_gen2(L,C,&(e.first)), i,i.size()+1);
    }
};

void AprioriGen(const cHashtree& L, cHashtree& C)
{
    for_each(L,Apriori_gen1(&L,&C));
}

/************************ IncrCount **************************/
//Function object to perform an operation during traversal
struct IncrCount : public unary_function<cItemset&, void>
{
    void operator() (cItemset& e) { e.second++; }
};

/************************ ResetCounts **************************/

//Function object to perform an operation during traversal
struct ResetCounts : public unary_function<cItemset&, void>
{
    ResetCounts() { }
    void operator() (cItemset& e) {e.second=0;}
};

/************************ IsPresent **************************/

//Function object to perform an operation during traversal
struct IsPresent : public unary_function<cItemset&, void>
{
    cHashtree& H1;
    cHashtree& H2;
    static bool ret;

    IsPresent(cHashtree& h1, cHashtree& h2) : H1(h1), H2(h2)
	    { ret = true; }
    void operator() (cItemset& e)
    {
	cHashtree::iterator i;
	i = H1.find(e.first);
	if (i != H1.end())
	{
	    e.second = i->second;
	    return;
	}
	i = H2.find(e.first);
	if (i != H2.end())
	{
	    e.second = i->second;
	    return;
	}
	ret = false;
    }

    operator bool() const { return ret; } //see section 11.4 of
    					 //Stroustrup, 3ed.
};

/************************ SetdbCount *************************/

//Function object to perform an operation during traversal
struct SetdbCount : public unary_function<cItemset&, void>
{
    cHashtree *h;

    SetdbCount(cHashtree& H) : h(&H) {}

    void operator() (cItemset& e)
    {
	cHashtree::iterator i = h->find(e.first);
	e.second = i->second - e.second;
    }
};

/************************ GetCount ***************************/

//Function object to perform an operation during traversal
struct GetCount : public unary_function<cItemset&, void>
{
    cHashtree *h;

    GetCount(cHashtree& H) : h(&H) {}

    void operator() (cItemset& e)
    {
	cHashtree::iterator i = h->find(e.first);
	if (i != h->end())
	    e.second = i->second;
    }
};

/************************ AddCounts **************************/

//Function object to perform an operation during traversal
struct AddCounts : public unary_function<cItemset&, void>
{
    cHashtree *h;
  
    AddCounts(cHashtree& H) : h(&H) {}

    void operator() (cItemset& e)
    {
	cHashtree::iterator i = h->find(e.first);
	if (i != h->end())
	    e.second += i->second;
    }
};

/************************ Difference **************************/

//Function object to perform an operation during traversal
struct Difference : public unary_function<cItemset&, void>
{
    cHashtree& H1;
    cHashtree& H2;

    Difference(cHashtree& h1, cHashtree& h2) : H1(h1), H2(h2) { }
    void operator() (cItemset& e)
    {
	if (! H1.contains(e.first))
	    H2.insert(e);
    }
};

/************************* MoveLarge **************************/

//Function object to perform an operation during traversal
struct MoveLarge : public unary_function<cItemset&, void>
{
    cHashtree *L;
    int minCount;
    MoveLarge(cHashtree &h, int m)
    	{ L = &h; minCount = m; }
    void operator() (cItemset& e)
    {
	if (e.second >= minCount)
	    L->move(e);
    }
};

/*
// BEGIN ADDITIONS FOR MASK
//Function object to copy cItemsets from cHashtree to
// vHashtree as vItemsets



struct Shift : public unary_function<cItemset&, void>
{
  cHashtree *C1;
  // cHashtree *C2;
  int pass;
  Shift(vHashtree &h,int p)
  { C1 = &h; pass = p; }
  void operator() (cItemset& e)
    {
      vItemset newEntry;  
      newEntry.first = e.first;
      vector<int> counts(pass+1);
      newEntry.second = counts;
      C1->move(newEntry);
    }
};


//Function object to increase approp elem corresponding to partial count
struct IncrAppropCount : public unary_function<vItemset&, void>{
  Itemset i;
  IncrAppropCount(Itemset &i1)
    	{ i = i1;}
    void operator() (vItemset& e) { 
      Itemset diff = i - e.first;
      int pc = i.size();
      int pc1 = diff.size();
      e.second[pc - pc1]++;   
    }  
};


//Function object to increase approp elem corresponding to partial count
// argument is boolean string
struct IncrAppropCount1 : public unary_function<vItemset&, void>{
  char* c;
  IncrAppropCount1(char* c1)
    	{ c = c1;}
    void operator() (vItemset& e) { 
      int count = 0;
      for (Itemset::const_iterator j = e.first.begin(); j != e.first.end(); j++)
	count = count + c[j->id] -48;
      e.second[count]++;
    }  
};




//Function object for correcting counts(reconstructing)
struct Correction : public unary_function<cItemset&, void>
{
  cHashtree *C;
  double* weights;
  int pass;
  double realcount;
    Correction(cHashtree &h,int ps,double* w)
    	{ C = &h; weights = w; pass = ps;}
    void operator() (vItemset& e){
      realcount = 0;
      for ( int i = 0; i <= pass; i++ )
	realcount = realcount + weights[i]*e.second[i];
      cHashtree::iterator j = C->find(e.first);
      j->second = ((int)realcount);
    }
};



*/
// END ADDITIONS FOR MASK

/******************Additions by SHIPRA***********************/


int reconstruct(cItemset& e,cHashtree* CMaskptr,int n,int dbrows,double** MCoeff,int mincount);

struct Print : public unary_function<cItemset&, void>
{
    //cItemsetBag *B;
    //Move(cItemsetBag &b) { B = &b; }
	//
    Print(){;}
    void operator() (cItemset& e)
    {
	//B->push_back(make_pair(Itemset(),e.second));
	//(B->back()).first.swap(e.first);
	//print("\n%d\n",e.second);
	
	//e type == cItemSet == pair<Itemset,int>
	//e.first type == Itemset == svector<Item>  
	    
	    cout<<e.first<<"\n";
    }
	
};

struct Reconstruct : public unary_function<cItemset&, void>
{
    cHashtree* CMaskptr;
    int n;
    double** MCoeff;
    cHashtree* Lptr;
    int mincount;
    int dbrows;
    
    Reconstruct(cHashtree* CM,cHashtree* Lp,int pass,int dbSize,double** Mat, int minc)
	    //Make CMask,L as reference param, passs mincount
    	{
		CMaskptr=CM;
		n=pass;
		MCoeff=Mat;
		Lptr=Lp;
		mincount=minc;
		dbrows=dbSize;
	}

    
    void operator() (cItemset& e)
    {
	
	//finding distorted counts of other combs using distorted count of 1111..n times
	//FIXME at present calculating 000000.. also for verifying,, remove later
	// i is no. of 1s in comb
	
	int SEst = reconstruct(e, CMaskptr, n, dbrows, MCoeff, mincount);
	
       /*int SEst=e.second * MCoeff[n];
	    
	*for(int i=0;i<n;i++)
	*{
	*	//Prefix1 has i 1s
	*	unsigned int Prefix1=pow(2,i)-1;	    
	*	//Suffix0 has n-i 1s
	*	unsigned int Suffix0=pow(2,n-i)-1;
	*    	
	*	//special cond for 00000..
	*	int Countcomb= (Prefix1==0) ? dbrows : 0; 
	*
	*	//Add count of 11111..n times, corresponds to subSuffix0 == Suffix0 
	*	Countcomb += pow(-1,n-i) * e.second;
	*
	*	for(int subSuffix0=0; subSuffix0 < Suffix0; subSuffix0++)
	*	{
	*		Prefix1<<n-i;
	*		
	*		int subcomb = (Prefix | subSuffix0);
	*		Itemset tmpItem = GetItemset(subcomb,e.first);
	*		
	*		//GetCount from CMask
	*		Countcomb += pow(-1,findOne(subSuffix0)) * Getcount(tmpItem,CMaskptr);
	*	}
	*
	*	SEst += Countcomb * find_nCr(n,i) * MCoeff[i];
	*}
	*
	* 
	*
	* //Check the SEst here itself and insert entry with distorted count in CMask and optionally correct estimated count in L
	*
	* if(SEst >= mincount)
	*	CMaskptr->insert(e);
	*
	* // count Reconstructed
	* e.second=SEst;
	*/

		
	//Move Large to L
	if(SEst >= mincount)
		Lptr->move(e);
	return;
	
	
    }
	
};

/***********************************************************************************/
//Get itemset corresponding to a subset of an itemset    
Itemset GetItemset(unsigned int subitem_bitset,Itemset i,int n)
    {
	   
	    Itemset subitem;// = new Itemset; 	
	    //cout<<i<<"\n";
	    
	    for (Itemset::const_iterator j = i.begin(); j != i.end(); j++)
	    {
		    if((subitem_bitset & (unsigned int)pow(2,n-1)) == 0)
		    {
		    	n--;
			continue;
		    }
		    else
		    {
		    	subitem.insert(Item(j->id));
	        	n--;
		    }
	    }
	    
	    
	    return(subitem);	   
    }

Itemset Rearrange_Itemset(Itemset i, int guide,int n)
{
	Itemset newItemset;// = new Itemset;
	Itemset suffix;// = new Itemset;
	
	//making prefix and suffix
	for (Itemset::const_iterator j = i.begin(); j != i.end(); j++)
	{
		//0 bit	
		if((guide & (unsigned int)pow(2,n-1)) == 0)
		    {
			suffix.push_back(Item(j->id));    
		    	n--;
		    }
		//1 bit
		else{
			newItemset.push_back(Item(j->id));
	        	n--;
		    }

	}

	//combining
	for (Itemset::const_iterator j = suffix.begin(); j != suffix.end(); j++)
	{
		newItemset.push_back(Item(j->id));
	}

	//cout<<"\nrearranged "<<i<<"="<<newItemset<<" by guide = "<<guide<<"\n";

	return(newItemset);
}


int Getcount(Itemset i,cHashtree* Treeptr)
    {
	int count;
	cHashtree::const_iterator j = Treeptr->find(i);
	count = j->second;
	return(count);	
    }

int findOnes(unsigned int x)
{
	int res=0;
	while(x>0)
	{
		res+=x % 2; 
		x=x/2;
	}
	return(res);
	
}

int find_nCr(int n, int r)
{
	if(r==0)
		return 1;
	if(n==r)
		return 1;
	return(find_nCr(n-1,r-1)+find_nCr(n-1,r));
}

int max(int x, int y)
{
	if(x>y)
		return(x);
	else
		return(y);

}
int min(int x, int y)
{
	if(x<y)
		return(x);
	else
		return(y);
}


int reconstruct(cItemset& e,cHashtree* CMaskptr,int n,int dbrows, double** MCoeff, int mincount)
{
	
	//printf("\n1\n");
	
	double SEst= e.second * MCoeff[0][0];
	
	if(DEBUG){cout<<"\nReconstructing count for item "<<e.first<<"\n";}
	if(DEBUG){cout<<"e.second = "<<e.second<<", Initialized SEst to "<<SEst<<"\n";}
	
	
	//for(int i=n-1;i>=0;i--)
	for(int x=(int)pow(2,n)-2;x>=0;x--)
	{
		
		int i=findOnes(x);
		Itemset efirst_shuf = Rearrange_Itemset(e.first,x,n);
		
	//printf("\n2\n");
	
		if(DEBUG){cout<<"\nrearranged "<<e.first<<"="<<efirst_shuf<<" by guide = "<<x<<"\n";}
		
		/*Finding distorted counts for combination having i 1s*/
		
		 //Prefix1 has i 1s
		 unsigned int Prefix1=(int)pow(2,i)-1;	   
		 //Suffix0 has n-i 1s
		 unsigned int Suffix0=(int)pow(2,n-i)-1;
		
		/*
		 * //Prefix1 has n-i 1s
		 * unsigned int Prefix1=(int)pow(2,n-i)-1;	   
		 * //Suffix0 has i 1s
		 * unsigned int Suffix0=(int)pow(2,i)-1;
		 */
		
	   	 
		if(DEBUG){fprintf(stdout,"\n\tPREFIX1 = %x SUFFIX0 = %x\n",Prefix1,Suffix0);}

		
		//special cond for 00000..
		int Countcomb= (Prefix1==0) ? dbrows : 0; 
	
		if(DEBUG){cout<<"\n\t\tCountcomb = "<<Countcomb;}
		
		//Add count of 11111..n times, corresponds to subSuffix0 == Suffix0 
		Countcomb += (int)pow(-1,n-i) * e.second;
		
		if(DEBUG){cout<<"\n\t\tCountcomb = "<<Countcomb;}
		
		Prefix1 = Prefix1<<n-i;
		
		for(int subSuffix0=0; subSuffix0 < Suffix0; subSuffix0++)
		{
			
			int subcomb = (Prefix1 | subSuffix0);
			if(subcomb==0)
				continue;
			
			//Itemset tmpItem = GetItemset(subcomb,e.first,n);
			Itemset tmpItem = GetItemset(subcomb,efirst_shuf,n);
			
			if(DEBUG){cout<<"\n\t\tcomb itemset "<<tmpItem;}
			if(DEBUG){cout<<" count = "<<Getcount(tmpItem,CMaskptr)<<"\n";}
			
			//GetCount from CMask
			Countcomb += (int)pow(-1,findOnes(subSuffix0)) * Getcount(tmpItem,CMaskptr);
			
			if(DEBUG){cout<<"\n\t\tCountcomb = "<<Countcomb;}
		}
		
	
		
	//printf("\n3\n");
		//SEst += Countcomb * MCoeff[n-i];	
		//XXX FOR TEST SCHEME
		//SEst += Countcomb * MCoeff[0][(int)pow(2,n)-1-x];	
		SEst += Countcomb * MCoeff[0][n-i];/*To reduce size of matrix*/
		
		
		if(DEBUG){cout<<"\n\tSEst = "<<SEst;}
		
	
	}
	/*/Check the SEst here itself and insert entry with distorted count in CMask and optionally correct estimated count in L
	 * */
	if((int)SEst >= mincount)
		CMaskptr->insert(e);
	
	// count Reconstructed
	
	//printf("\n4\n");
	
	if(DEBUG){cout<<"count reconstructed for "<<e.first<<"\nreconstructed count = "<<SEst<<"\n";}
	e.second=(int)SEst;

	return ((int)SEst);
}
	


/***********************************************************************************/




double** UpdateMatrix(double** PrevMat, double** &PrevdistMat, int pass, double p, double q)
    
{
	//double* newMatRow = new double[pass+1];
	//int n=(int)pow(2,pass);
	double** newMat; //= allocate_mat(n*n);
	
	//double** newDistMat = allocate_mat(n); //To reduce size of matrix
	double** newDistMat;	
	if(pass>1)
	{
	destroy(PrevMat,pass);
	}
	newDistMat=allocate_mat((pass+1));

	
	/*
	 * double q=1-p;
	 */
	
	
	
	 if(pass==1)
	 	{
	 		 /*newMatRow[0]=p/(p*p - q*q);
			 * newMatRow[1]=-q/(p*p - q*q);		
			 * //PrevMatRow=newMatRow;
			 * if(DEBUG){cout<<"Updated matrix row for pass "<<pass<<"\n";}
			 * for(int i=0;i<=pass;i++)
			 * 	cout<<newMatRow[i]<<" ";
			 * cout<<"\n";	
			 * return(newMatRow);
			 */

	
			
			newDistMat[0][0]=p;
			newDistMat[0][1]=1-q;	
			newDistMat[1][0]=1-p;
			newDistMat[1][1]=q;	
		}
	else
		/*{	
		int nprev=(int)pow(2,pass-1);
		for(int i=0;i<nprev;i++)
			for(int j=0;j<nprev;j++)
			{
				newDistMat[i][j]		=p*PrevdistMat[i][j];
				newDistMat[i][j+nprev]		=(1-q)*PrevdistMat[i][j];
				newDistMat[i+nprev][j]		=(1-p)*PrevdistMat[i][j];
				newDistMat[i+nprev][j+nprev]	=q*PrevdistMat[i][j];
				
			}
		
		destroy(PrevdistMat,nprev);
		destroy(PrevMat,nprev);
		}*/
		{
		for(int i=0;i<pass+1;i++)
			for(int j=0;j<pass+1;j++)
			{
				newDistMat[pass-j][pass-i]=0;
				for(int t=max(0,i+j-pass);t<=min(i,j);t++)
				{
					newDistMat[pass-j][pass-i]+=find_nCr(i,t)*pow(p,t)*pow(1-p,i-t)*find_nCr(pass-i,j-t)*pow(1-q,j-t)*pow(q,pass+t-i-j);
				
				}
				
			
			}		
			
		}

	
	//PrevdistMat=newDistMat;
	
	
	//cout<<"Distortion matrix for pass \n"<<pass<<"\n";
	//printmat(newDistMat,n);
	
	/*matrix* mat = new matrix(newDistMat,n);
	 * matrix* newMatClass = mat->inv();
	 * newMat = newMatClass->values();
	 * delete(mat);
	 */ 
	
		
	newMat=my_invert(newDistMat,pass+1);
	
	destroy(newDistMat,pass+1);
	

	//cout<<"Inverted matrix for pass "<<pass<<"\n";
	//printmat(newMat,n);
	
	
	/*PrevMatRow=newMatRow;	
	 *for(int i=0;i<n;i++)
	 * {
	 * for(int j=0;j<n;j++)
	 * 	cout<<newMat[i]<<" ";
	 * cout<<"\n";
	 * }
	 * cout<<"\n";
	 *
	 * PrevMat=newMat;
	 */
	
	return(newMat);
}


/************************* Move to HashTree**************************/

//Function object to perform an operation during traversal
/*struct Move : public unary_function<cItemset&, void>
{
    cItemsetBag *B;
    Move(cItemsetBag &b) { B = &b; }
    void operator() (cItemset& e)
    {
	B->push_back(make_pair(Itemset(),e.second));
	(B->back()).first.swap(e.first);
    }
};*/


/******************END Additions by SHIPRA***********************/




/************************* InsertLarge **************************/

//Function object to perform an operation during traversal
struct InsertLarge : public unary_function<cItemset&, void>
{
    cHashtree *L;
    int minCount;
    InsertLarge(cHashtree &h, int m)
    	{ L = &h; minCount = m; }
    void operator() (cItemset& e)
    {
	if (e.second >= minCount)
	    L->insert(e);
    }
};

/************************* InsertSmall **************************/

//Function object to perform an operation during traversal
struct InsertSmall : public unary_function<cItemset&, void>
{
    cHashtree *L;
    int minCount;
    InsertSmall(cHashtree &h, int m)
    	{ L = &h; minCount = m; }
    void operator() (cItemset& e)
    {
	if (e.second < minCount)
	    L->insert(e);
    }
};

/********************** MoveLargeSmall ***********************/

//Function object to perform an operation during traversal
struct MoveLargeSmall : public unary_function<cItemset&, void>
{
    cHashtree *L;
    cHashtree *N;
    int minCount;
    MoveLargeSmall(cHashtree& h1, cHashtree& h2, int m) :
	    L(&h1), N(&h2), minCount(m) { }
    void operator() (cItemset& e)
    {
	if (e.second >= minCount)
	    L->move(e);
	else
	    N->move(e);
    }
};

/********************** InsertLargeSmall ***********************/

//Function object to perform an operation during traversal
struct InsertLargeSmall : public unary_function<cItemset&, void>
{
    cHashtree *L;
    cHashtree *N;
    int minCount;
    InsertLargeSmall(cHashtree& h1, cHashtree& h2, int m) :
	    L(&h1), N(&h2), minCount(m) { }
    void operator() (cItemset& e)
    {
	if (e.second >= minCount)
	    L->insert(e);
	else
	    N->insert(e);
    }
};

/************************* Move **************************/

//Function object to perform an operation during traversal
struct Move : public unary_function<cItemset&, void>
{
    cItemsetBag *B;
    Move(cItemsetBag &b) { B = &b; }
    void operator() (cItemset& e)
    {
	B->push_back(make_pair(Itemset(),e.second));
	(B->back()).first.swap(e.first);
    }
};

/************************* Insert **************************/

//Function object to perform an operation during traversal
struct Insert : public unary_function<cItemset&, void>
{
    cHashtree *h;
    Insert(cHashtree &H) : h(&H) { }
    void operator() (cItemset& e) { h->insert(e); }
};

/************************* Remove **************************/

//Function object to perform an operation during traversal
struct Remove : public unary_function<cItemset&, void>
{
    cHashtree *h;
    Remove(cHashtree &H) : h(&H) { }
    void operator() (cItemset& e) { h->remove(e.first); }
};

/********************** Move into a tree ********************/

//Function object to perform an operation during traversal
struct MoveTree : public unary_function<cItemset&, void>
{
    cHashtree *h;
    MoveTree(cHashtree &H) : h(&H) { }
    void operator() (cItemset& e) { h->move(e); }
};

/************************* Find Items ***********************/

//Function object to perform an operation during traversal
struct FindItems : public unary_function<cItemset&, void>
{
    vector<bool> *v;
    FindItems(vector<bool> &V) : v(&V) { }
    void operator() (cItemset& e)
    {
	Itemset::iterator i;
	for (i = e.first.begin(); i != e.first.end(); i++)
	    (*v)[i->id] = true;
    }
};

/************** Get Itemsets of Particular Depth ************/

//Function object to perform an operation during traversal
struct GetDepth : public unary_function<cItemset&, void>
{
    AncesTable *t;
    cHashtree *h;
    int d;
    GetDepth(AncesTable &T, cHashtree &H, int D) : t(&T), h(&H), d(D) {}
    void operator() (cItemset& e)
    {
	if (t->depth(e.first) == d)
	    h->insert(e);
    }
};

/********** Get Descendants of a Particular Itemset *********/

//Function object to perform an operation during traversal
struct GetDescendants1 : public unary_function<cItemset&, void>
{
    AncesTable *t;
    cHashtree *h;
    Itemset *i;
    GetDescendants1(AncesTable &T, cHashtree &H, Itemset &I) : 
	    t(&T), h(&H), i(&I) {}
    void operator() (cItemset& e)
    {
	if (t->isAncestor(*i, e.first))
	    h->insert(e);
    }
};

//Function object to perform an operation during traversal
struct GetDescendants : public unary_function<cItemset&, void>
{
    AncesTable *t;
    cHashtree *h;
    cHashtree *r;
    GetDescendants(AncesTable &T, cHashtree &H, cHashtree &result) : 
	    t(&T), h(&H), r(&result) {}
    void operator() (cItemset& e)
    {
	for_each(*h, GetDescendants1(*t, *r, e.first), e.first.size());
    }
};

#endif
