/**********************************************************************
    Copyright (C) 2004 Database Systems Lab, Supercomputer Education and
    Research Centre, Indian Institute of Science, Bangalore, INDIA.
    http://dsl.serc.iisc.ernet.in

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***********************************************************************/


/***********************************************************************
 The AprioriGen Function and other functions useful for Apriori like
 algorithms.
 
 We use the functor concept of C++ and multiple types of traversal
 routines to accomplish the tasks.  We have routines to traverse
 hashtrees -- either all itemsets in a hashtree or subsets of a given
 itemset or supersets of it.  These traversal routines are defined in
 hashTraverse.h which is included from hashtree.h.

 As expected, to define a functor, we need to define all the functions
 and functors that it uses, before the actual code of this functor. 
 However we think in a top-down approach here.  Therefore to understand
 the code of each functor, read each section from bottom up.
***********************************************************************/
#ifndef APRIORI_H_
#define APRIORI_H_

#include "cItemset.h"
#include "taxonomy.h"

/************************ AprioriGen *****************************/

//Function object to perform an operation during traversal
struct Apriori_gen2 : public unary_function<const cItemset&, void>
{
    const cHashtree *L;
    cHashtree *C;
    const Itemset *i1;
    static long no_pruned;

    Apriori_gen2(const cHashtree *l, cHashtree *c,const Itemset *i)
	    { L=l; C=c; i1=i; }

    void operator() (const cItemset& e)
    {
	/* preconditions:
	   1) e.first which is an itemset, must have size>=1
	   2) e.first.size() == i1->size() is true */

	const Itemset& i2 = e.first;
	if (equal(i1->begin(),(i1->end())-1,i2.begin()) &&
		i1->back() < i2.back())
	{
	    Itemset candidate;
	    candidate.reserve(i1->size()+1);
	    candidate.insert(candidate.begin(),i1->begin(),i1->end());
	    candidate.push_back(i2.back());

	    //prune
	    if ( i1->size() > 1 ) {
		Itemset::iterator j;
		for ( j=candidate.begin(); j<candidate.end()-2; j++ ) {
		    Item temp = *j;
		    candidate.erase(j);
		    if ( ! L->contains(candidate) )
		    {
			no_pruned++;
			return;
		    }
		    candidate.insert(j,temp);
		}
	    }

	    cItemset newEntry;
	    newEntry.first.swap(candidate);
	    newEntry.second = 0;
	    C->move(newEntry);
	}
    }
};

//Function object to perform an operation during traversal
struct Apriori_gen1 : public unary_function<const cItemset&, void>
{
    const cHashtree *L;
    cHashtree *C;

    Apriori_gen1(const cHashtree *l, cHashtree *c)
	    { L=l; C=c; }

    void operator() (const cItemset& e)
    {
	//precondition: e.first must have size >= 1

	Itemset i(e.first.begin(), e.first.end()-1);
	for_each_extension(*L, Apriori_gen2(L,C,&(e.first)), i,i.size()+1);
    }
};

void AprioriGen(const cHashtree& L, cHashtree& C)
{
    for_each(L,Apriori_gen1(&L,&C));
}

/************************ IncrCount **************************/
//Function object to perform an operation during traversal
struct IncrCount : public unary_function<cItemset&, void>
{
    void operator() (cItemset& e) { e.second++; }
};

/************************ ResetCounts **************************/

//Function object to perform an operation during traversal
struct ResetCounts : public unary_function<cItemset&, void>
{
    ResetCounts() { }
    void operator() (cItemset& e) {e.second=0;}
};

/************************ IsPresent **************************/

//Function object to perform an operation during traversal
struct IsPresent : public unary_function<cItemset&, void>
{
    cHashtree& H1;
    cHashtree& H2;
    static bool ret;

    IsPresent(cHashtree& h1, cHashtree& h2) : H1(h1), H2(h2)
	    { ret = true; }
    void operator() (cItemset& e)
    {
	cHashtree::iterator i;
	i = H1.find(e.first);
	if (i != H1.end())
	{
	    e.second = i->second;
	    return;
	}
	i = H2.find(e.first);
	if (i != H2.end())
	{
	    e.second = i->second;
	    return;
	}
	ret = false;
    }

    operator bool() const { return ret; } //see section 11.4 of
    					 //Stroustrup, 3ed.
};

/************************ SetdbCount *************************/

//Function object to perform an operation during traversal
struct SetdbCount : public unary_function<cItemset&, void>
{
    cHashtree *h;

    SetdbCount(cHashtree& H) : h(&H) {}

    void operator() (cItemset& e)
    {
	cHashtree::iterator i = h->find(e.first);
	e.second = i->second - e.second;
    }
};

/************************ GetCount ***************************/

//Function object to perform an operation during traversal
struct GetCount : public unary_function<cItemset&, void>
{
    cHashtree *h;

    GetCount(cHashtree& H) : h(&H) {}

    void operator() (cItemset& e)
    {
	cHashtree::iterator i = h->find(e.first);
	if (i != h->end())
	    e.second = i->second;
    }
};

/************************ AddCounts **************************/

//Function object to perform an operation during traversal
struct AddCounts : public unary_function<cItemset&, void>
{
    cHashtree *h;

    AddCounts(cHashtree& H) : h(&H) {}

    void operator() (cItemset& e)
    {
	cHashtree::iterator i = h->find(e.first);
	if (i != h->end())
	    e.second += i->second;
    }
};

/************************ Difference **************************/

//Function object to perform an operation during traversal
struct Difference : public unary_function<cItemset&, void>
{
    cHashtree& H1;
    cHashtree& H2;

    Difference(cHashtree& h1, cHashtree& h2) : H1(h1), H2(h2) { }
    void operator() (cItemset& e)
    {
	if (! H1.contains(e.first))
	    H2.insert(e);
    }
};

/************************* MoveLarge **************************/

//Function object to perform an operation during traversal
struct MoveLarge : public unary_function<cItemset&, void>
{
    cHashtree *L;
    int minCount;
    MoveLarge(cHashtree &h, int m)
    	{ L = &h; minCount = m; }
    void operator() (cItemset& e)
    {
	if (e.second >= minCount)
	    L->move(e);
    }
};

/************************* InsertLarge **************************/

//Function object to perform an operation during traversal
struct InsertLarge : public unary_function<cItemset&, void>
{
    cHashtree *L;
    int minCount;
    InsertLarge(cHashtree &h, int m)
    	{ L = &h; minCount = m; }
    void operator() (cItemset& e)
    {
	if (e.second >= minCount)
	    L->insert(e);
    }
};

/************************* InsertSmall **************************/

//Function object to perform an operation during traversal
struct InsertSmall : public unary_function<cItemset&, void>
{
    cHashtree *L;
    int minCount;
    InsertSmall(cHashtree &h, int m)
    	{ L = &h; minCount = m; }
    void operator() (cItemset& e)
    {
	if (e.second < minCount)
	    L->insert(e);
    }
};

/********************** MoveLargeSmall ***********************/

//Function object to perform an operation during traversal
struct MoveLargeSmall : public unary_function<cItemset&, void>
{
    cHashtree *L;
    cHashtree *N;
    int minCount;
    MoveLargeSmall(cHashtree& h1, cHashtree& h2, int m) :
	    L(&h1), N(&h2), minCount(m) { }
    void operator() (cItemset& e)
    {
	if (e.second >= minCount)
	    L->move(e);
	else
	    N->move(e);
    }
};

/********************** InsertLargeSmall ***********************/

//Function object to perform an operation during traversal
struct InsertLargeSmall : public unary_function<cItemset&, void>
{
    cHashtree *L;
    cHashtree *N;
    int minCount;
    InsertLargeSmall(cHashtree& h1, cHashtree& h2, int m) :
	    L(&h1), N(&h2), minCount(m) { }
    void operator() (cItemset& e)
    {
	if (e.second >= minCount)
	    L->insert(e);
	else
	    N->insert(e);
    }
};

/************************* Move **************************/

//Function object to perform an operation during traversal
struct Move : public unary_function<cItemset&, void>
{
    cItemsetBag *B;
    Move(cItemsetBag &b) { B = &b; }
    void operator() (cItemset& e)
    {
	B->push_back(make_pair(Itemset(),e.second));
	(B->back()).first.swap(e.first);
    }
};

/************************* Insert **************************/

//Function object to perform an operation during traversal
struct Insert : public unary_function<cItemset&, void>
{
    cHashtree *h;
    Insert(cHashtree &H) : h(&H) { }
    void operator() (cItemset& e) { h->insert(e); }
};

/************************* Remove **************************/

//Function object to perform an operation during traversal
struct Remove : public unary_function<cItemset&, void>
{
    cHashtree *h;
    Remove(cHashtree &H) : h(&H) { }
    void operator() (cItemset& e) { h->remove(e.first); }
};

/********************** Move into a tree ********************/

//Function object to perform an operation during traversal
struct MoveTree : public unary_function<cItemset&, void>
{
    cHashtree *h;
    MoveTree(cHashtree &H) : h(&H) { }
    void operator() (cItemset& e) { h->move(e); }
};

/************************* Find Items ***********************/

//Function object to perform an operation during traversal
struct FindItems : public unary_function<cItemset&, void>
{
    vector<bool> *v;
    FindItems(vector<bool> &V) : v(&V) { }
    void operator() (cItemset& e)
    {
	Itemset::iterator i;
	for (i = e.first.begin(); i != e.first.end(); i++)
	    (*v)[i->id] = true;
    }
};

/************** Get Itemsets of Particular Depth ************/

//Function object to perform an operation during traversal
struct GetDepth : public unary_function<cItemset&, void>
{
    AncesTable *t;
    cHashtree *h;
    int d;
    GetDepth(AncesTable &T, cHashtree &H, int D) : t(&T), h(&H), d(D) {}
    void operator() (cItemset& e)
    {
	if (t->depth(e.first) == d)
	    h->insert(e);
    }
};

/********** Get Descendants of a Particular Itemset *********/

//Function object to perform an operation during traversal
struct GetDescendants1 : public unary_function<cItemset&, void>
{
    AncesTable *t;
    cHashtree *h;
    Itemset *i;
    GetDescendants1(AncesTable &T, cHashtree &H, Itemset &I) : 
	    t(&T), h(&H), i(&I) {}
    void operator() (cItemset& e)
    {
	if (t->isAncestor(*i, e.first))
	    h->insert(e);
    }
};

//Function object to perform an operation during traversal
struct GetDescendants : public unary_function<cItemset&, void>
{
    AncesTable *t;
    cHashtree *h;
    cHashtree *r;
    GetDescendants(AncesTable &T, cHashtree &H, cHashtree &result) : 
	    t(&T), h(&H), r(&result) {}
    void operator() (cItemset& e)
    {
	for_each(*h, GetDescendants1(*t, *r, e.first), e.first.size());
    }
};

#endif
