/**********************************************************************
    Copyright (C) 2004 Database Systems Lab, Supercomputer Education and
    Research Centre, Indian Institute of Science, Bangalore, INDIA.
    http://dsl.serc.iisc.ernet.in

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***********************************************************************/


/********************* Interface to the Database ***********************
 AUTHOR: Vikram Pudi

 DESCRIPTION:

 As far as we are concerned till now, a database must just contain (in
 some form) a sequence of itemsets.

 An itemset is a set of items.  An item is an elementary entity that is
 not further defined, except that it comes from a predefined set of
 possible entities.

 It is assumed that the items in an itemset are in ascending order.  An
 itemset is implemented as a vector of items, as defined in the
 standard template library (stl) of C++.

 Item ids are assumed to start from 0 till some max-1.  Even though we
 do not use this fact here, it will be used later.  It is the job of
 this file to convert items to ids from 0 to max-1, if they are not
 already of that form in the database.

 Along with each database file, there must be another file which
 contains information about the database, such as number of items,
 tuples, etc.  We call this file as a metafile.  The syntax of this
 file is as follows --

	pathname of the database file
	number of items
	number of tuples

 An alternative syntax, which is useful for incremental mining
 algorithms is --

	pathname of the database file
	number of items
	number of tuples in part 1
	number of tuples in part 2
	.
	.
	.
 
 There can be more than one metafile per database file.  This is useful
 for comparing incremental algorithms versus non-incremental ones.

 The database is designed as a class "Database" and behaves like an
 input stream of itemsets.  To open the database, we pass a string
 containing the pathname of the metafile to the constructor.  For the
 alternative syntax of the metafile as defined above, we may also pass
 which part of the database we want, as a second argument to the
 constructor.  To open the first part, we have part=1.  To open the
 second part (increment), we can have part=2, etc.

 Since a database behaves like an input stream of itemsets, we can have
 code like --

 	Itemset i;  Database db;
	db >> i;

 The operation "Database >>" returns the database, which can then be
 tested to see if any more input operations can be done. The state can
 be tested merely by using the name of the database. So the following
 will perform a pass over the database --

	while (db >> i)
 	{
	    //use i
	}
***********************************************************************/

#ifndef DATABASE_H_
#define DATABASE_H_

#include <fstream.h>
#include "item.h"
#include "taxonomy.h"

const int BUFSIZE = 8092 * sizeof(Item);

class Database {
    ifstream db; //the actual database file
    char buf[BUFSIZE];
    int noAttributes; //no of items
    int databaseSize; //no of tuples
    streampos start; //start of 1st tuple
    int cursor;
    AncesTable table; //for generalized rules

public :
    Database(char *metadata,int);
	//metadata is name of a file containing information about data
        //the int will tell which part of the database file is needed
        //1 is for the 1st part, etc. ... useful for incremental mining

    int size() const { return databaseSize; }
    int rows() const { return size(); }
    int columns() const { return noAttributes; }
    AncesTable& tax() { return table; }

    void rewind() { db.seekg(start); db.clear(); cursor = 0; } //goto first tuple
    void skip(); //skip the next tuple
    friend Database& operator>>(Database& d, Itemset& i);
    	//operator gets the next tuple and puts it into itemset i
    friend Database& operator>>(Database& d, char* &c);
    // it puts the whole line in c
    operator bool() { return (db.good() && cursor <= size()); }
    	//returns true if the input stream is good(), i.e. there is
    	//input to be seen. See sections 7.3.2 and 10.3.2 of
    	//Stroustrup 2nd edition.
        //Should have been a const function. But good() is not
    	//implemented as a const function.. so
};

Database::Database(char *metadata, int part = 1 )
{
    ifstream mf(metadata);
    if (! mf)
    {
	cout << "Couldn't open metafile: " << metadata << endl;
	exit(1);
    }

    char datafile[256];
    mf >> datafile;
    mf >> noAttributes;
    if (! mf)
    {
	mf.clear();
	char taxfile[256];
	mf >> taxfile;
	Taxonomy taxonomy(taxfile);
	table.construct(taxonomy);
	mf >> noAttributes;
    }


// instead of bin file....we are openiing ascii file
    /*
#ifdef __GNUC__
    db.open(datafile, ios::in|ios::binary);
    // OPEN AS ASCII FILE

#else //SPARCompiler doesn't support this
    db.open(datafile);
#endif

*/
    db.open(datafile);


    if (! db)
    {
	cout << "Couldn't open database file: " << datafile << endl;
	exit(1);
    }

    db.setbuf(buf, BUFSIZE);

    for (int n = 0; n < part; n++)
    {
	mf >> databaseSize;
	cursor = 0;
	if (n < part-1)
	    for (int count = 0; count < databaseSize; count++)
		skip();
    }
    start = db.tellg();
}

inline Database& operator>>(Database& d, Itemset& i)
{
    if (++d.cursor > d.size())
	return d;

    // EXTRA ARGUMENT GIVEN
    i.database_read(d.db,d.columns());
    return d;
}


inline Database& operator>>(Database& d, char* &c)
{
    if (++d.cursor > d.size())
	return d;
    //delete [] c;
    c = new char[d.columns()];
    (d.db).read(c,sizeof(char)*d.columns());
    //    cout<<c<<"from db.h \n";
    return d;
}



void Database::skip()
{
    if (++cursor > size())
	return;

    int j,noItems;
    db.read((char*)&j,sizeof(j)); //skip transaction number
    db.read((char*)&j,sizeof(j)); //skip customer number
    db.read((char*)&noItems,sizeof(noItems));
    db.seekg(noItems, ios::cur);
}

inline void write_big_endian(ofstream &fp, int stuff)
{
    char buf[sizeof(int)];
    for (int i = 0; i < (int)sizeof(int); i++)
	buf[sizeof(int)-i-1] = (stuff & (0x000000ff<<(8*i))) >> (8*i);
    fp.write(buf, sizeof(int));
}

void writeTransaction(Itemset& i, int tid, ofstream &fp)
{
    int cid = tid; //customer id
    int nitems = i.size();

    fp.write((char *)&cid, sizeof(int));
    fp.write((char *)&tid, sizeof(int));
    fp.write((char *)&nitems, sizeof(int));
    for (int j = 0; j < nitems; j++)
    {
	int item = i[j].id;
	fp.write((char *)&item, sizeof(int));
    }
    if (! fp)
    {
	cerr << "Cannot write to output file!" << endl;
	exit(1);
    }
}

void writeTransactionBigEndian(Itemset& i, int tid, ofstream &fp)
{
    int cid = tid; //customer id
    int nitems = i.size();

    write_big_endian(fp, cid);
    write_big_endian(fp, tid);
    write_big_endian(fp, nitems);
    for (int j = 0; j < nitems; j++)
    {
	int item = i[j].id;
	write_big_endian(fp, item);
    }
    if (! fp)
    {
	cerr << "Cannot write to output file!" << endl;
	exit(1);
    }
}

#endif
