/**********************************************************************
    Copyright (C) 2004 Database Systems Lab, Supercomputer Education and
    Research Centre, Indian Institute of Science, Bangalore, INDIA.
    http://dsl.serc.iisc.ernet.in

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***********************************************************************/


/***********************************************************************
 This file contains some routines that may be used to generate random
 numbers of various distributions.  It was first implemented for an
 assignment in a Performance Modelling course.  It may be useful to
 implement the Sampling and other randomized algorithms.

 Each distribution is implemented as a class.  If x is an object of
 some distribution, then x.getNext() will return the next random number
 in that distribution.  Each object will need to be initialized by a
 seed.  The range of the random number generator is divided into some
 number of periods, and the start of each period corresponds to a seed. 
 These seeds are stored in an instance of class SeedBank.  This instance
 must therefore be defined first and only then can we declare any
 distribution objects.
***********************************************************************/
#ifndef RANDOM_H_
#define RANDOM_H_

#define PI 3.141592654

#include <stdlib.h> //for rand_r()
#include <string.h> //for memcpy()
#include <iostream.h> //for cout..but this can be got rid of
#include <math.h> //for log(), cos(), sqrt()

extern "C" int rand_r(unsigned int *); //such a declaration is necessary

typedef unsigned int seed_t;

class SeedBank //only one instance of this class may be defined
{
    unsigned int noPeriods;
    unsigned int period;
    seed_t startSeed;
    seed_t *seeds;
    unsigned int nextSeedIndex;

public:

    SeedBank( unsigned int numPeriods, int periodLen, seed_t startseed = 1 )
    {
	noPeriods = numPeriods;
	period = periodLen;
	startSeed = startseed;
	nextSeedIndex = 0;

	seed_t seed;
	unsigned int maxPeriod = noPeriods * period;
	seeds = new seed_t[noPeriods];
	unsigned int index = 0;

	seed = startSeed;
	for ( int i = 1; i <= maxPeriod; i++ )
	{
	    if ( i % period == 1 )
		seeds[index++] = seed;
	    seed = rand_r(&startSeed);
	}

	cout << endl << "Seeds: ";
	for ( i = 0; i < noPeriods; i++ ) cout << seeds[i] << " ";
	cout << endl;
	cout.flush ( );
    }

    seed_t *getSeed( void ) { return &seeds[nextSeedIndex++]; }
    unsigned int getPeriod( void ) { return period; }
    unsigned int getMin( void ) { return 0; } //depends on rand_r()
    unsigned int getMax( void ) { return 32767; } //depends on rand_r()
    unsigned int getRange( void ) { return ( getMax( ) - getMin( ) + 1 ); }
};

class Distr
{

public:

    virtual float getNext( void ) = 0;
    float oops( void );
    //nothing here.
};

class Distribution : public Distr
{
protected:

    seed_t *seed;
    SeedBank *seedBank;

public:

    Distribution( SeedBank *bank )
    {
	seedBank = bank;
	seed = seedBank->getSeed( );
    }

    float getNext( void ) { return ((float)rand_r( seed )); }
    float getNextFloat( void ) //aaargh! overloading? compiler gives error if we do so.
    {
	return ( (rand_r( seed ) + 1) / (float) seedBank->getRange( ) );
    }
};

class Gaussian : public Distr
{
protected:

    seed_t *seed1, *seed2;
    SeedBank *seedBank;
    float mean, variance;

public:

    Gaussian( SeedBank *bank, float mu, float var )
    {
	mean = mu;
	variance = var;
	seedBank = bank;
	seed1 = seedBank->getSeed( );
	seed2 = seedBank->getSeed( );
    }

    float getNext( void )
    {
	float u1 = (rand_r(seed1)+1)/(float)seedBank->getRange();
	float u2 = (rand_r(seed2)+1)/(float)seedBank->getRange();

	float x1 = mean + sqrt(variance) * cos(2*PI*u1) * sqrt(-2*log(u2));
/*	float x2 = mean + sqrt(variance) * sin(2*PI*u1) * sqrt(-2*log(u2));
		---x2 is another independant gaussian distribution */

	return x1;
    }
};

class DiscreteUniform : public Distribution
{
protected:

    int min, max;
    unsigned int range;

public:

    DiscreteUniform( SeedBank *bank, int lower, int higher )
	: Distribution( bank )
    {
	min = lower;
	max = higher;
	range = max - min + 1;
    }

    float getNext( void )
    {
	return ((float)(( min + (int)(Distribution::getNext( )) % range )));
    }
};

class Uniform : public Distribution
{
protected:

    float min, max, range;

public:

    Uniform( SeedBank *bank, float lower, float higher )
	: Distribution( bank )
    {
	min = lower;
	max = higher;
	range = max - min;
    }

    float getNext( void )
    {
	return ( min + Distribution::getNextFloat( ) * range );
    }
};

class ContinuousUniform : public Distribution
{
public:

    ContinuousUniform( SeedBank *bank )
	: Distribution( bank ) {}

    float getNext( void )
    {
	return Distribution::getNextFloat( );
    }
};

class RandVar : public ContinuousUniform //make this protected instead?
{
protected:

    int noOutcomes;
    float *pmf;

public:

    RandVar( SeedBank *bank, int numOutcomes, float *probVect )
	: ContinuousUniform( bank )
    {
	noOutcomes = numOutcomes;
	pmf = new float[noOutcomes];
	memcpy( pmf, probVect, noOutcomes*sizeof (float) );
    }

    float getNext( void )
    {
        float next = ContinuousUniform::getNext( );
	float cumulative;
	
	cumulative = 0;
	for ( int i = 0; i < noOutcomes; i++ )
	{
	    cumulative += pmf[i];
	    if (next < cumulative)
		return i;
	}

	return ((float)i);
    }
};

class Expo : public ContinuousUniform //make this protected instead?
{
protected:

    float lambda;

public:

    Expo( SeedBank *bank, float mean )
	: ContinuousUniform( bank )
    {
	lambda = 1/mean;
    }

    float getNext( void )
    {
        float retVal = ContinuousUniform::getNext( );
	return ( lambda * log( 1 / retVal ) );
    }
};

class DiscreteExpo : public Expo //make this protected instead?
{
protected:

    int min, max;
    unsigned int range;

public:

    DiscreteExpo( SeedBank *bank, float parameter, int low, int high )
	: Expo( bank, parameter )
    {
	min = low;
	max = high;
	range = max - min + 1;
    }

    float getNext( void )
    {
        float retVal = Expo::getNext( );
	return ((float)( min + retVal * range ));
    }
};

#endif
