#ifndef _ALLOCATOR_H_
#define _ALLOCATOR_H_

/***********************************************************************
 AUTHOR: Srikanta Bedathur
 DESCRIPTION:
    Template for Allocating the nodes through the Storage Manager.

    Copyright (C) 2004 Database Systems Lab, Supercomputer Education and
    Research Centre, Indian Institute of Science, Bangalore, INDIA.
                     http://dsl.serc.iisc.ernet.in

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***********************************************************************/


/*********************************************************************
 * 
 * The new node allocations are done in this class.
 * These are singletons for each internal and leaf nodes
 * The Fileids are [0...4), Pageids [1...MAXBLOCKS) and 
 * Recids are [0...pagecapacity]
 *
 * This is also the place to keep track of number of "extents" of 
 * each class of pages.
 *********************************************************************/

template <class A, bool isleaf>
class Allocator {

private:
  StorageManager <A,isleaf>* sm;

private:
  SFXULONG _remsize; //remaining size in the page
  double _totaldepth;

public:
  page_t* active_page;
  SFXULONG active_recid;
  SFXULONG active_pageid;
  SFXULONG active_fileid;


public:
  ////////////////////////////////////////////////
  ///  Constructor
  ////////////////////////////////////////////////
  Allocator (StorageManager<A,isleaf> * _sm) {
    ASSERT (_sm);
    
    sm = _sm;

    active_page = new page_t();

    ASSERT (active_page);

    active_fileid = 0;
    active_pageid = 1;
    active_recid = 0;

#ifdef GEOMETRIC
    _totaldepth = 1;
#else
    _totaldepth = 0;
#endif
    _remsize = active_page -> datasize();

    sm->pin (active_fileid, 
	     active_pageid,
	     active_page);
  }
  
  //////////////////////////////////////////////
  /// Destructor
  /////////////////////////////////////////////
  ~Allocator () {
    sm -> unpin (active_fileid, 
		 active_pageid);

    /**
     * Send info about number of extents,
     * number of pages in the last extent
     * to the options/metainfo class
     **/

  }


  //////////////////////////////////////////////
  // Create a new node - with necessary check to
  // move to a new page if needed. Returning a
  // Reference to the new node.
  //////////////////////////////////////////////
  Ref newnode (SFXULONG depth) {
    /* 
     * If there is no space in the current 
     * page, then reset the current page, and
     * move to new page
     */
    if (_remsize < sizeof (A)){
      _resetPage();
    }

#ifdef GEOMETRIC
    _totaldepth *= depth;
#else
    _totaldepth += depth;
#endif

    SFXULONG offset = active_recid * sizeof (A);
    active_recid++;
    //    active_page->count();

    _remsize -= sizeof (A);

    return ( Ref ( (A*)(active_page->data() + offset),
		   (active_recid - 1),
		   active_pageid,
		   active_fileid,
		   isleaf));
  }
		 
private:
  void _resetPage() {
    DBG_TRACE("_resetPage" << endl);
#ifdef GEOMETRIC
    geom_resetpage();
#else
    arith_resetpage();
#endif
    
    sm -> unpin (active_fileid, active_pageid);
    active_page = new page_t ();
    active_pageid ++;

    if (active_pageid > MAXBLOCKS) {
      active_pageid = 1;
      active_fileid += 1;
    }

    active_recid = 0;
    _remsize = active_page -> datasize();

    sm -> pin (active_fileid, active_pageid, active_page);
  }
  
  void geom_resetpage() {

    double temp = (double) 1.0 / (double)active_recid;
    ASSERT (temp > 0);
    if (isinf (_totaldepth))
      _totaldepth = MAXDOUBLE;

    if (isleaf) {
      active_page->avgdepth ((double)1.0 / (pow(_totaldepth, temp)));
    }
    else {
      active_page->avgdepth (pow(_totaldepth, temp));
    }

    if (errno == EDOM) {
      cerr << "FAILURE : EDOM error set! " << endl;
      exit (-10);
    }
    _totaldepth = 1;
  }

  void arith_resetpage() {
    if (isleaf){
      active_page -> avgdepth ((double)active_recid / _totaldepth);
    }
    else {
      active_page -> avgdepth (_totaldepth /(double)active_recid);
    }
    _totaldepth = 0;
  }
};

#endif
