#ifndef _S_MAN_TOPQ_H_
#define _S_MAN_TOPQ_H_

/***********************************************************************
 AUTHOR: Srikanta Bedathur
 DESCRIPTION:
    Storage Manager extension with TOPQ buffering. By settting the queue
    length to 0, it reduces to TOP buffering.

    Copyright (C) 2004 Database Systems Lab, Supercomputer Education and
    Research Centre, Indian Institute of Science, Bangalore, INDIA.
                     http://dsl.serc.iisc.ernet.in

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***********************************************************************/
#include "s_man.h"
#include "storage.h"

using namespace std;
#if(__GNUC__==3)
using namespace __gnu_cxx;
#endif

/* Comparison functor for the heap*/
struct heap_comp {
  bool operator () (const pagerec_t* one,
		    const pagerec_t* two) const {
    return (one -> page_ptr -> avgdepth() <
	    two -> page_ptr -> avgdepth() );
  }
};

/* heap */
typedef priority_queue <pagerec_t*, vector<pagerec_t*>, heap_comp> page_heap;


template <class A, bool isleaf>
class TopqStorageManager : public StorageManager <A, isleaf> {

protected:
  
  page_heap buffer_heap;
  vector <pagerec_t*> AinQ;

  long MAXHEAP;
  long Qlen;

  SFXULONG Ain_size;

public:
  TopqStorageManager (Storage *storage,
		      long buffpoolsize, 
		      long _qlen, 
		      enum PHASE _phase = CONSTRUCT) :
    StorageManager <A, isleaf> (storage, buffpoolsize, _phase) {
    
    MAXHEAP = buffpoolsize;
    Qlen = _qlen;

    MAXBUFFER = MAXHEAP + Qlen;
    Ain_size = 0;
  }

protected:
  void makespace () {
    /* If there is enough space in the heap, then nothing
     * to do, just return
     */
    if (buffer_heap.size() < MAXHEAP) {
      return;
    }
    
    vector <pagerec_t*> temp;

    /***
     * Locate the shallowest page which doesn't have
     * any active references to it. And store the 
     * candidate you get in the way - which are shallower
     * but have active references pointing to them
     ***/
    while ((buffer_heap.top()) -> ref_count > 0) {
      temp.push_back (buffer_heap. top());
      buffer_heap.pop();
    }

    Ain_insert (buffer_heap.top());
    buffer_heap.pop();

    if (Ain_size > Qlen) {
      Ain_remove ();
    }

    ASSERT (Ain_size + buffer_heap.size() <= MAXBUFFER);

    for (vector<pagerec_t*>::iterator iter = temp.begin();
	 iter != temp.end();
	 iter++) {
      buffer_heap.push ((*iter));
    }
    return;
  }

  void insert (pagerec_t* page) {
    buffer_heap.push (page);
    return;
  }
  
  void reposition(pagerec_t* freshpage) {
    return;
  }


  void Ain_insert (pagerec_t* inpage) {
    ASSERT (inpage);
    AinQ.push_back (inpage);
    Ain_size++;
  }

  void Ain_remove (){
    vector <pagerec_t*>::iterator iter = AinQ.begin();
    while ((*iter) -> ref_count > 0 && iter != AinQ.end()) {
      iter++;
    }
    
    ASSERT (iter != AinQ.end());
    
    purge (*iter);
    AinQ.erase (iter);
    AinQ.resize (Qlen);
    Ain_size--;
  }

  void cleanup () {

    
    AinQ.clear();
    Ain_size = 0;
    while (!buffer_heap.empty()) {
      buffer_heap.pop();
    }
    StorageManager<A,isleaf>::cleanup();
  }
    
};

#endif

