#ifndef _DISK_STORAGE_H_
#define _DISK_STORAGE_H_

/***********************************************************************
 AUTHOR: Srikanta Bedathur
 DESCRIPTION:
    Disk-based storage class - extension of abstract Storage class.

    Copyright (C) 2004 Database Systems Lab, Supercomputer Education and
    Research Centre, Indian Institute of Science, Bangalore, INDIA.
                     http://dsl.serc.iisc.ernet.in

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***********************************************************************/
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>


#include <string>

#include "options.h"
#include "defines.h"
#include "page.h"
#include "malloc.h"
#include "storage.h"

using namespace std;

class DiskStorage : public Storage{
private:
  int fd[4];
  int flags;
  int mode;
  int write_count [4];
public:

  ////////////////
  /// Constructor 
  /// Default flags are for construction
  ////////////////
  DiskStorage (bool _isleaf, enum PHASE _phase = CONSTRUCT) : Storage(_isleaf,_phase) {
    if (phase == CONSTRUCT || phase == BOTH) {
      flags = O_RDWR | O_CREAT | O_TRUNC;
      mode = S_IRWXU;
    }
    if (phase == SEARCH) {
      flags = O_RDONLY;
      mode = S_IRWXU;
    }
    fd [0] = fd[1] = fd[2] = fd[3] = -1;
  }
  ////////////////
  /// Destructor
  ////////////////
  ~DiskStorage () {
    for (int i = 0;i < 4; i++) {
      if (fd[i] > 0) {
	fdatasync (fd[i]);
	close (fd[i]);
      }
    }
  }

  //////////////////
  // Open the given fileid - if not already open.
  // Base File name is obtained from options
  // object.
  //////////////////
  int s_open (SFXULONG fileid) {
    if (fd [fileid] < 0) {
      string temp;
      ostringstream ostr;
      if (isleaf) {
	ostr << Options::getLeafBasefile () << "." << fileid;
      }
      else {
	ostr << Options::getInternalBasefile () << "." << fileid;
      }
      temp = ostr.str();
      
      fd [fileid] = open (temp.c_str(),
			  flags,
			  mode);
      if (fd[fileid] < 0) 
	err_msg ("Couldn't open the file ", DISASTER);
    }
    return fd [fileid];
  }

  ///////////////////
  // Write to the given fileid, at the specified offset
  ///////////////////
  
  int s_pwrite (SFXULONG fileid,
		SFXULONG pageid,
		page_t* page_ptr) {
    ASSERT (phase != SEARCH);
    int sk = 0;
    if ((sk = lseek (fd [fileid],
		     (pageid - 1) * DISKPAGESIZE,
		     SEEK_SET)) < 0)
      err_msg ("Couldn't seek to write", DISASTER);

    int w = 0;
    if ((w = write (fd[fileid],
		    page_ptr -> page(),
		    DISKPAGESIZE)) < DISKPAGESIZE)
      err_msg ("Couldn't write at the position", DISASTER);

#ifdef FILESTATS
    if(isleaf) {
      write_leaf_count++;
    }
    else {
      write_internal_count++;
    }
#endif
    write_count[fileid]++;

    if (write_count [fileid] == 1000){
      fdatasync (fd[fileid]);
      write_count [fileid]=0;
    }
    return w;
  }

  //////////////////////////
  // Read from the given fileid.
  //////////////////////////
  int s_pread (SFXULONG fileid,
	       SFXULONG pageid,
	       page_t* page_ptr) {
    
    int sk = 0;

    if (fd [fileid] == -1) 
      s_open (fileid);

    if ((sk = lseek (fd[fileid],
		     (pageid - 1) * DISKPAGESIZE,
		     SEEK_SET)) < 0) 
      err_msg ("Couldn't seek to read", DISASTER);
    
    int r = 0;
    if ((r = read (fd[fileid],
		   page_ptr -> page(),
		   DISKPAGESIZE)) < DISKPAGESIZE)
      err_msg ("Couldn't read from the position", DISASTER);

#ifdef FILESTATS
    if (isleaf) {
      read_leaf_count++;
    }
    else {
      read_internal_count++;
    }
#endif
    return r;
  }
};


#endif
