/***********************************************************************
 AUTHOR: Srikanta Bedathur
 DESCRIPTION:
    References that shield disk/buffer accesses from the algorithm.

    Copyright (C) 2004 Database Systems Lab, Supercomputer Education and
    Research Centre, Indian Institute of Science, Bangalore, INDIA.
                     http://dsl.serc.iisc.ernet.in

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***********************************************************************/
#include <assert.h>
#include "smartref.h"
#include "defines.h"

//Constructors
//
//1. Given a "pointer", we construct a unreferenced Ref
//
//NOTE: Remember to keep this consistent with operator=(long) below
Ref::Ref(const SFXULONG _pointer){
  _pinned = false;
#ifndef SMALL_PAGE  
  SFXULONG temp = _pointer;
  _recid = temp & 0x03ff; temp >>= 10;
  _blockid = temp & 0x07ffff; temp >>= 19;
  _fileid = temp & 0x03; temp >>= 2;
  _isleaf = temp;
#else
  SFXULONG temp = _pointer;
  _recid = temp & 0x03; temp >>= 2;
  _blockid = temp & 0x0effffff; temp >>= 27;
  _fileid = temp & 0x03; temp >>= 2;
  _isleaf = temp;
#endif
  _intpointee = NULL;
}
//
//2. Copy Constructor -- Copy from another Ref object
//
Ref::Ref(const Ref& rhs){
  _isleaf = rhs._isleaf;
  _fileid = rhs._fileid;
  _blockid = rhs._blockid;
  _recid = rhs._recid;

  if (isleaf())
    _leafpointee = rhs._leafpointee;
  else
    _intpointee = rhs._intpointee;

  _pinned = rhs._pinned;

  if (_pinned){
    ASSERT(_intpointee);
    incrementref();//increment the reference count of the block
  }
}
//
//3. "construct" the Ref bottom up. Give the data and build the Ref
//
Ref::Ref(SfxNode* ptr,
	 SFXULONG recid,
	 SFXULONG blockid,
	 SFXULONG fileid,
	 bool isleaf) {
#ifndef SMALL_PAGE
  ASSERT (fileid < 4);
  ASSERT (blockid < (1 << 19));
  ASSERT (recid < (1 << 10));
#else
  ASSERT (fileid < 4);
  ASSERT (blockid < (1 << 27));
  ASSERT (recid < (1 << 2));
#endif
  
  _isleaf = (isleaf)?1:0;
  _fileid = fileid;
  _blockid = blockid;
  _recid = recid;

  if(ptr) {
    _pinned = true;
    incrementref(); //increment the reference count
  }
  else 
    _pinned = false;

  if(this->isleaf())
    _leafpointee = ptr;
  else
    _intpointee = (SfxInternalNode*)ptr;
}



////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////

// SOME OPERATORS

//1. Assignment to a "pointer" -- behaves almost like constructor (1)
Ref& Ref::operator=(SFXULONG rhs){
  if (_pinned)
    decrementref();//current blockid ref-count decrements
#ifndef SMALL_PAGE
  SFXULONG temp = rhs;
  _recid = temp & 0x03ff; temp >>= 10;
  _blockid = temp & 0x07ffff; temp >>= 19;
  _fileid = temp & 0x03; temp >>= 2;
  _isleaf = temp;
#else
  SFXULONG temp = rhs;
  _recid = temp & 0x03; temp >>= 2;
  _blockid = temp & 0x0effffff; temp >>= 27;
  _fileid = temp & 0x03; temp >>= 2;
  _isleaf = temp;
#endif
  _pinned = false;
  _intpointee = NULL;

  return (*this);
}

//2. Assignment to another Ref. 
Ref& Ref::operator=(const Ref& rhs){
  if (_pinned)
    decrementref();

  _isleaf = rhs._isleaf;
  _fileid = rhs._fileid;
  _blockid = rhs._blockid;
  _recid = rhs._recid;
  
  if(isleaf())
    _leafpointee = rhs._leafpointee;
  else
    _intpointee = rhs._intpointee;

  _pinned = rhs._pinned;

  if (_pinned) {
    ASSERT (_intpointee);
    incrementref(); //increment the refernces to the block
  }
  return *this;
}

// 3. CAST INTO A POINTER
Ref::operator SFXULONG() const {
  SFXULONG retval = 0;
#ifndef SMALL_PAGE
  retval = _isleaf; retval <<= 2;
  retval |= _fileid; retval <<= 19;
  retval |= _blockid; retval <<= 10;
  retval |= _recid;
#else
  retval = _isleaf; retval <<= 2;
  retval |= _fileid; retval <<= 27;
  retval |= _blockid; retval <<= 2;
  retval |= _recid;
#endif
  return retval;
}

// The following two operators are result of some kludging that 
// we are following now. Since Ref wrappers both internal nodes 
// and also the leaf nodes, we need to some way of differentiating
// the dereferencing of these two types. But we have only one -> 
// operator! So the kludge.. (Ugggh...)


/// To obtain the SfxNode format.
const SfxNode* Ref::operator->() {
  DBG_TRACE ("Ref::operator->() " << _pinned << ", " << isleaf() << " - " <<  _fileid << ", " << _blockid << ", " << _recid << endl);
  if (!_pinned) 
    requestobject();
  if (isleaf())
    return _leafpointee;
  return (SfxNode*)_intpointee;
}

/// To get the internal node -- if there is one
const SfxInternalNode* Ref::operator* () {
  if (isleaf()) 
    return NULL;
  if (!_pinned)
    requestobject();
  return _intpointee;
}


//////////////////////////////////////////////////////////////////////
/// Destructor
//////////////////////////////////////////////////////////////////////
Ref::~Ref () {
  if (_pinned)
    decrementref();
  _pinned = false;
  _blockid = 0;
  _recid = 0;
  _fileid = 0;
  _isleaf = 0;
}


// To get a writeable reference from this
WRef Ref::update() {
  if (!_pinned)
    requestobject();
  return WRef(*this);
}


///////////////////////////////////////////////////////////////////////
/// WRef operators
///////////////////////////////////////////////////////////////////////
SfxNode* WRef::operator->() {
//   if (!_pinned) 
//     requestobject();
  if (isleaf())
    return _leafpointee;
  return (SfxNode*)_intpointee;
}

SfxInternalNode* WRef::operator* () {
  if (isleaf()) 
    return NULL;
//   if (!_pinned)
//     requestobject();
  return _intpointee;
}


WRef::~WRef () {
//   decrementref(); //increment has happened in the construction.
//   _pinned = false; 
//   _blockid = _recid = 0;
//   _fileid = 0;
//   _isleaf = 0;
}
