// File:  heap.C 
// Created by: Pradeep Shenoy, (purdy@cse.iitb.ernet.in)
// Last modified: 19 Aug 1999
//
// Description: 
// 	Heap that stores ($snakeID$, $TID$) pairs heaped on the TID
// attribute.  The use for this code: In the counting of second-level
// candidate itemsets and in the DAG merge step. 
// 	The process involves putting the first entries of each snake in the
// heap, consecutively removing the smallest TID, performing count updates
// due to presence of this snake (itemset) in this Txn, and entering its
// next presence in the heap.

#ifndef _tidheap_h
#define _tidheap_h

#include  <iostream.h>

struct heapbuc_t {
    int readid;
    int TID;

    // List of next-level itemsets whose counts to be updated.
    Cand_t **updatelist;
    int updatecount;
};

class  TidHeap {
    
    heapbuc_t *heap;	// Heap implemented as an array
    
    heapbuc_t tmp;
    heapbuc_t DELVAL;
    
    int heapsize;
    int ndata;
    
    inline void TidHeap::heapify_up(void){
	
	int index = ndata-1; 
	int parent = (index-1)/2;
	while (index && heap[parent].TID > heap[index].TID){
    	    tmp = heap[parent];
	    heap[parent] = heap[index];
	    heap[index] = tmp;
	    
    	    index = parent; parent = (index-1)/2;
	}
	return;
    }
    
    inline void TidHeap::heapify_down(void){
	
	int done = 0;	
	int wrong = 0;  // Root is (poss) wrongly placed.
	int lchild, rchild, min;
	
	while (!done){
	    
	    min = wrong;
	    lchild = 2*wrong+1;
	    rchild = 2*wrong+2;
	    
	    if (lchild < ndata && heap[min].TID > heap[lchild].TID)
		min = lchild;
	    if (rchild < ndata && heap[min].TID > heap[rchild].TID)
		min = rchild;
	    
	    if (min == wrong) done = 1;
	    else {
		tmp = heap[min];
		heap[min] = heap[wrong]; heap[wrong] = tmp; 
		
		wrong = min;
	    }
	    
	}
	return;
    }
    
public:
    TidHeap(int maxsize){
	heapsize = maxsize; ndata = 0;
	heap = new heapbuc_t[heapsize+1];
	
    	for (int i = 0;i <= heapsize;i++)
	    heap[i].TID = heap[i].readid = 0;
    }
    ~TidHeap(void) {if (heap) delete [] heap;}
	
	
	heapbuc_t TidHeap::deleteMIN(void){
	    
	    if (!ndata){        // Empty heap -- error condition
		tmp.TID = -1;
		return tmp;
	    }
	    
	    tmp = heap[0];
	    heap[0] = heap[--ndata]; 
	    heap[ndata] = tmp;
	    
	    heapify_down();
	    
	    return heap[ndata];
	}
    
    int TidHeap::insert(int readid, int TID, Cand_t **update=NULL, 
	    	int nupdates=0){
	
	if (ndata >= heapsize) return -1; // Overflow
	
	heap[ndata].readid = readid;
	heap[ndata].TID = TID;
	heap[ndata].updatelist = update;
	heap[ndata].updatecount = nupdates;
	
	++ndata;
	heapify_up(); 
	return ndata;
    }
    
};
#endif // #ifndef _tidheap_h
